using System;

namespace pdc
{
	public class PDCDevice: HIDDevice
	{
		private volatile PDCInputReport inputReport = null;
        private volatile bool bRemoved = false;

		public PDCDevice() { }
		public override InputReport CreateInputReport() { return new PDCInputReport(this);	}
        protected override void HandleDataReceived(InputReport oInRep) { inputReport = (PDCInputReport)oInRep; }
        protected override void HandleDeviceRemoved() { bRemoved = true; }
        protected override void HandleDeviceArrived() { bRemoved = false; }

		public PDCInputReport ReadReport()
		{
            if (bRemoved) return null;

			// Wait for an input report to arrive (busy waiting - bad...)
			inputReport = null;
			BeginAsyncRead();

            for (int Count = 1; (Count < 10) && (inputReport == null); Count++)
			    System.Threading.Thread.Sleep(100);

            return inputReport;
		}

		public void WriteReport(PDCOutputReport outputReport)
		{
            if (bRemoved) return;

			try
			{
				// Reset the input report so the next call to ReadReport has new data
				inputReport = null;

				// Write data
				Write(outputReport);
			}
			catch(HIDDeviceException)
			{
				this.Dispose(); // Device may have been removed!
				
			}
		}

		/// <summary>
		/// Finds the PDC USB Receiver.
		/// Caution: VID 0000 and PID 0002 looks suspicious.
		/// </summary>
		/// <returns>A new PDCDevice or null if not found.</returns>
		public static PDCDevice FindPDCDevice(byte vendor, byte product)
		{
			// VID and PID for PDC device are 0x0000 and 0x0002, respectively
			return (PDCDevice)FindDevice(vendor, product, typeof(PDCDevice));
		}

	}

	public class PDCInputReport: InputReport
	{
		public PDCInputReport(HIDDevice dev): base(dev) {}

		// We don't parse anything here at the moment
		public override void ProcessData()
		{
		}
	}

	public class PDCOutputReport: OutputReport
	{
		public PDCOutputReport(HIDDevice dev): base(dev) {}
	}
}
