/* ==================================================================================================
 * Park Distance Control (PDC) plugin
 * File: pdc.cs
 * Author: Dipl.-Ing. Wolfgang Otto
 * eMail: Wolfgang@exner-otto.de
 * Date: August 2008
 * THIS PLUGIN IS A CONVERSION OF THE MARVELESS TOOL OF
 * **************************************************************************************************
 * Roberto Montanari, RoadRunner PDC Plugin															*
 * His VB sources are converted to C# and modified to be used as a CENTRAFUSE plugin				*
 * Thanks for his great work!																		*
 * **************************************************************************************************
 * This file is subject to the condition of GNU GENERAL PUBLIC LICENSE								*
 * Therefore you are free to use this software for your own requirements and use the				*
 * source code for any modification.																*	
 * The only wish is to inform the author about any kind of development.								*
 * =================================================================================================*/

    // to use the simulation mode change the complilation directive to SIMULATION
    #define _SIMULATION         // remove _ to activate
    #define _SPECIALFOLDER      // remove _ to activate

using System;
using System.Drawing;
using System.Windows.Forms;
using System.Xml;
using System.IO;
using System.Data;
using System.Data.OleDb;
using System.Web;
using System.Collections;
using System.Threading;
using System.Text;
using System.ComponentModel;
using System.Runtime.InteropServices;
using System.Drawing.Drawing2D;
using System.Diagnostics;
using System.Reflection;
using DirectShowLib;
using centrafuse.Plugins;

using System.Collections.Generic;

namespace pdc
{	

	public class pdc : CFPlugin
    {
        #region Global variables
        // a small enum to record the graph state
        public enum PlayState {Stopped, Paused, Running, Init};

        // Application-defined message to notify app of filtergraph events
        public const int WM_GRAPHNOTIFY = 0x8000 + 1;
                
        // sound handling during buzzer sound output
        public bool bAUDIO_STOP = false;
        public bool ENABLE_AUDIO_STOP = true;

        public bool BuzzerFrontStatus;
        public bool BuzzerRearStatus;
        public struct VolumeLevelStruct
        {
            public bool IsReduced;
            public float LastVolume;
        }

        public VolumeLevelStruct VolumeLevel;
        
        //========================================================================================
        /*
         * Information:
         * If the parameter useDebug is true then a lot of functions writes information
         * into a log file called pdc.log inside plugin path.
         * 
         * All errors detected by [try..catch] functions are routed to the global error.log
         * file of Centrafuse. This output can not be stopped by configurable parameter, because
         * it is a serious problem.
         */
        public static bool useDebug = false;
        public static bool useErrorOutput = false;
        //========================================================================================

        public static bool bCLOSEABLE = true;
        public static int GHoldOpenTime = 5 * 1000; // setup parameter HOLDOPENTIME

        #endregion Global variables

        #region Distance specific variables
        public enum Unit { meters = 0, inches = 1, feet = 2 }
        public static Unit DistanceUnit;
	    public static string DistanceSuffix;
        #endregion
        		
		#region Video specific variables

        public static IBaseFilter VideoDevice = null;
        IVideoWindow  videoWindow = null;
        IMediaControl mediaControl = null;
        IMediaEventEx mediaEventEx = null;
        IGraphBuilder graphBuilder = null;
        ICaptureGraphBuilder2 captureGraphBuilder = null;
        PlayState currentState = PlayState.Stopped;
        DsROTEntry rot = null;
        public static bool bFullScreen = false;             // boolean value for indication full video mode
        public static bool bHeaderWritten = false;          // Date/Time Information header for error log file
        
        #endregion


        #region PDC USB Device and ID declaration
        
        public static PDCDevice deviceRear = null;
        public static PDCDevice deviceFront = null;

		public const byte VendorID          = 0x0;
		public const byte ProductIDRear     = 0x2;
        public const byte ProductIDFront    = 0x3;

        private volatile PDCOutputReport outputReport = null;
        private volatile PDCInputReport inputReport = null;
       
        public struct SensorData        // Structure for distance sensors
        {
            public byte Distance;
            public byte OldFrontValue;
            public byte OldRearValue;
        }
        public static SensorData[] Sensor = new SensorData[8];
        
        #endregion
        
        protected System.Windows.Forms.Timer checkTimer;
        protected System.Windows.Forms.Timer HoldOpenTimer;
        
        #if SIMULATION
        protected System.Windows.Forms.Timer SliderTimer;
        #endif
		
		#region PAGE and CONFIG handling
		public static int  LENLABEL5_8 = 30;
		public static int  LENLABEL1_4 = 40;
        public static string GLOBAL_SLIDER_PATH = string.Empty;
        public static byte SIMULATION_DISTANCE = 254;

        public const string PluginName = "PDC";
        public const string PluginPath = @"plugins\" + PluginName + @"\";
        public const string PluginPathSkins = PluginPath + @"Skins\";
        public const string PluginPathLanguages = PluginPath + @"Languages\";
        public const string PluginPathIcons = PluginPath + @"Icons\";
        public const string ConfigurationFile = "config.xml";
        public const string LogFile = PluginName + ".log";
        public static string LanguageSection = "/APPLANG/" + PluginName.ToUpper() + "/";

#if !(USEPLGPATH)
        public static string LogFilePath = CFTools.AppDataPath + @"\Plugins\" + PluginName + @"\" + LogFile;
        public static string pConfigurationFile = CFTools.AppDataPath + @"\Plugins\" + PluginName + @"\" + ConfigurationFile;
#else
        public static string LogFilePath = CFTools.StartupPath + @"\Plugins\" + PluginName + @"\" + LogFile;
        public static string pConfigurationFile = CFTools.StartupPath + @"\Plugins\" + PluginName + @"\" + ConfigurationFile;
#endif
		#endregion
				
        #region VIDEO CAPTURING
        //##############################################################################################################
      
        public void ChangePreviewState(bool showVideo)
        {
            int hr = 0;

            // If the media control interface isn't ready, don't call it
            if (this.mediaControl == null) CaptureVideo();

            if (showVideo)
            {
                if (this.currentState != PlayState.Running)
                {
                    // Start previewing video data
                    hr = this.mediaControl.Run();
                    this.currentState = PlayState.Running;
                    WriteLog("ChangePreviewState:PlayState = RUNNING");
                }
            }
            else
            {
                // Stop previewing video data
                hr = this.mediaControl.StopWhenReady();
                this.currentState = PlayState.Stopped;
                WriteLog("ChangePreviewState:PlayState = STOPPED");
            }
        }
        public void HandleGraphEvent()
        {
            int hr = 0;
            EventCode evCode;
            IntPtr evParam1, evParam2;

            if (this.mediaEventEx == null) return;

            while (this.mediaEventEx.GetEvent(out evCode, out evParam1, out evParam2, 0) == 0)
            {
                // Free event parameters to prevent memory leaks associated with
                // event parameter data.  While this application is not interested
                // in the received events, applications should always process them.
                hr = this.mediaEventEx.FreeEventParams(evCode, evParam1, evParam2);
                DsError.ThrowExceptionForHR(hr);

                // Insert event processing code here, if desired
            }
        }
        public void ResizeVideoWindow()
        {
            // Resize the video preview window to match owner window size
            if (this.videoWindow != null)
            {
                if (bFullScreen == false)
                    #region REDUCED VIDEOCREEN
                    this.videoWindow.SetWindowPosition(
                        (int)(Int32.Parse(pluginSkinReader.GetSkinAttribute("/SKIN/SECTIONS/SECTION/PDC/PICTUREBOXES/VIDEOFRAME/X")) * pluginSkinReader.skinWidthRatio),
                        (int)(Int32.Parse(pluginSkinReader.GetSkinAttribute("/SKIN/SECTIONS/SECTION/PDC/PICTUREBOXES/VIDEOFRAME/Y")) * pluginSkinReader.skinHeightRatio),
                        (int)(Int32.Parse(pluginSkinReader.GetSkinAttribute("/SKIN/SECTIONS/SECTION/PDC/PICTUREBOXES/VIDEOFRAME/WIDTH")) * pluginSkinReader.skinWidthRatio),
                        (int)(Int32.Parse(pluginSkinReader.GetSkinAttribute("/SKIN/SECTIONS/SECTION/PDC/PICTUREBOXES/VIDEOFRAME/HEIGHT")) * pluginSkinReader.skinHeightRatio)
                        );
                    #endregion
                else
                    #region FULLSCREEN
                    this.videoWindow.SetWindowPosition(
                        (int)(Int32.Parse(pluginSkinReader.GetSkinAttribute("/SKIN/SECTIONS/SECTION/PDC/PICTUREBOXES/VIDEOFRAMEFULL/X")) * pluginSkinReader.skinWidthRatio),
                        (int)(Int32.Parse(pluginSkinReader.GetSkinAttribute("/SKIN/SECTIONS/SECTION/PDC/PICTUREBOXES/VIDEOFRAMEFULL/Y")) * pluginSkinReader.skinHeightRatio),
                        (int)(Int32.Parse(pluginSkinReader.GetSkinAttribute("/SKIN/SECTIONS/SECTION/PDC/PICTUREBOXES/VIDEOFRAMEFULL/WIDTH")) * pluginSkinReader.skinWidthRatio),
                        (int)(Int32.Parse(pluginSkinReader.GetSkinAttribute("/SKIN/SECTIONS/SECTION/PDC/PICTUREBOXES/VIDEOFRAMEFULL/HEIGHT")) * pluginSkinReader.skinHeightRatio)
                        );
                    #endregion
            }
        }
        public void SetupVideoWindow()
        {
            int hr = 0;

            // Set the video window to be a child of the main window
            hr = this.videoWindow.put_Owner(this.Handle);
            DsError.ThrowExceptionForHR(hr);

            hr = this.videoWindow.put_WindowStyle(WindowStyle.Child | WindowStyle.ClipChildren);
            DsError.ThrowExceptionForHR(hr);

            // Use helper function to position video window in client rect 
            // of main application window
            ResizeVideoWindow();

            // Make the video window visible, now that it is properly positioned
            hr = this.videoWindow.put_Visible(OABool.True);
            DsError.ThrowExceptionForHR(hr);
        }
        public void CloseInterfaces()
        {
            // Stop previewing data
            if (this.mediaControl != null)
                this.mediaControl.StopWhenReady();

            this.currentState = PlayState.Stopped;

            // Stop receiving events
            if (this.mediaEventEx != null)
                this.mediaEventEx.SetNotifyWindow(IntPtr.Zero, WM_GRAPHNOTIFY, IntPtr.Zero);

            // Relinquish ownership (IMPORTANT!) of the video window.
            // Failing to call put_Owner can lead to assert failures within
            // the video renderer, as it still assumes that it has a valid
            // parent window.
            if (this.videoWindow != null)
            {
                this.videoWindow.put_Visible(OABool.False);
                this.videoWindow.put_Owner(IntPtr.Zero);
            }

            // Remove filter graph from the running object table
            if (rot != null)
            {
                rot.Dispose();
                rot = null;
            }

            // Release DirectShow interfaces
            Marshal.ReleaseComObject(this.mediaControl); this.mediaControl = null;
            Marshal.ReleaseComObject(this.mediaEventEx); this.mediaEventEx = null;
            Marshal.ReleaseComObject(this.videoWindow); this.videoWindow = null;
            Marshal.ReleaseComObject(this.graphBuilder); this.graphBuilder = null;
            Marshal.ReleaseComObject(this.captureGraphBuilder); this.captureGraphBuilder = null;
        }
        public void GetInterfaces()
        {
            int hr = 0;

            // An exception is thrown if cast fail
            this.graphBuilder = (IGraphBuilder)new FilterGraph();
            this.captureGraphBuilder = (ICaptureGraphBuilder2)new CaptureGraphBuilder2();
            this.mediaControl = (IMediaControl)this.graphBuilder;
            this.videoWindow = (IVideoWindow)this.graphBuilder;
            this.mediaEventEx = (IMediaEventEx)this.graphBuilder;

            hr = this.mediaEventEx.SetNotifyWindow(this.Handle, WM_GRAPHNOTIFY, IntPtr.Zero);
            DsError.ThrowExceptionForHR(hr);
        }
        private IBaseFilter CreateFilter(Guid category, string friendlyname)
        {
            object source = null;
            Guid iid = typeof(IBaseFilter).GUID;
            foreach (DsDevice device in DsDevice.GetDevicesOfCat(category))
            {
                if (device.Name.CompareTo(friendlyname) == 0)
                {
                    device.Mon.BindToObject(null, null, ref iid, out source);
                    break;
                }
            }

            return (IBaseFilter)source;
        }
        public void CaptureVideo()
        {
            int hr = 0;
            IBaseFilter sourceFilter = null;

            try
            {
                // Get DirectShow interfaces
                GetInterfaces();

                // Attach the filter graph to the capture graph
                hr = this.captureGraphBuilder.SetFiltergraph(this.graphBuilder);
                WriteLog("CaptureVideo:[Error code] captureGraphBuilder.SetFiltergraph: " + DsError.GetErrorText(hr)); 
                DsError.ThrowExceptionForHR(hr);

                // Use the system device enumerator and class enumerator to find
                // a video capture/preview device, such as a desktop USB video camera.
                string devicepath = this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/VIDEODEVICE");
                WriteLog("CaptureVideo: Devicepath = " + devicepath); 
                try
                {
                    pdc.VideoDevice = CreateFilter(FilterCategory.VideoInputDevice, devicepath);
                    sourceFilter = pdc.VideoDevice;

                    // Add Capture filter to our graph.
                    hr = this.graphBuilder.AddFilter(sourceFilter, "PDC");
                    WriteLog("CaptureVideo:[Error code] Addfilter: " + DsError.GetErrorText(hr)); 
                    DsError.ThrowExceptionForHR(hr);           

                    // Render the preview pin on the video capture filter
                    // Use this instead of this.graphBuilder.RenderFile
                    hr = this.captureGraphBuilder.RenderStream(PinCategory.Preview, MediaType.Video, sourceFilter, null, null);
                    WriteLog("CaptureVideo:[Error code] RenderStream: " + DsError.GetErrorText(hr)); 
                    DsError.ThrowExceptionForHR(hr);

                    // Now that the filter has been added to the graph and we have
                    // rendered its stream, we can release this reference to the filter.
                    WriteLog("CaptureVideo: BEFORE ReleaseObject"); 
                    Marshal.ReleaseComObject(sourceFilter);
                    WriteLog("CaptureVideo: AFTER ReleaseObject"); 

                    // Set video window style and position
                    WriteLog("CaptureVideo: BEFORE SetupVideoWindow"); 
                    SetupVideoWindow();
                    WriteLog("CaptureVideo: AFTER SetupVideoWindow"); 

                    // Add our graph to the running object table, which will allow
                    // the GraphEdit application to "spy" on our graph
                    WriteLog("CaptureVideo: BEFORE DsROTEntry"); 
                    rot = new DsROTEntry(this.graphBuilder);
                    WriteLog("CaptureVideo: AFTER DsROTEntry"); 
                    
                    // Start previewing video data
                    hr = this.mediaControl.Run();
                    WriteLog("CaptureVideo:[Error code] mediaControl.Run: " + DsError.GetErrorText(hr)); 
                    DsError.ThrowExceptionForHR(hr);

                    // Remember current state
                    this.currentState = PlayState.Running;
                    WriteLog("CaptureVideo: sucessfully started"); 
                }
                catch 
                { 
                    this.currentState = PlayState.Stopped;
                    WriteLog("CaptureVideo:[ERROR] CaptureVideo could not be started"); 
                }
                
            }
            catch (Exception errmsg) { CFTools.writeError(DsError.GetErrorText(hr), errmsg.StackTrace); }
        }

        //##############################################################################################################
        #endregion

        #region DISTANCE SYMBOL HANDLING

        public bool STOP_AUDIO
        {
            get { return bAUDIO_STOP;  }
            set 
            { 
                bAUDIO_STOP = value;
                if (bAUDIO_STOP == true)
                    if (ENABLE_AUDIO_STOP) this.CF_systemCommand(CF_Actions.MUTEAUDIO);
                else
                    this.CF_systemCommand(CF_Actions.UNMUTEAUDIO);
            }
        }
        internal byte GetSliderNumber(ref byte dist)
        {
            // Default (No Action)
            byte sldnr;
            if (dist >= 200) sldnr = 20;
            else             sldnr = Convert.ToByte(Math.Round((Convert.ToDecimal(dist) / 11), 0));
            if (sldnr < 0)   sldnr = 20;
            return sldnr;
        }
        public string DistanceToString(ref byte Distance)
        {
            string retStr = string.Empty;
            double d = Distance;

            if (Distance > 200)
            {
                retStr = "----" + DistanceSuffix;
            }
            else
            {
               switch (DistanceUnit)
                {
                    case Unit.meters:
                        retStr = String.Format("{0:F2} ", d /100) + DistanceSuffix;
                        break;
                    case Unit.inches:
                        retStr = String.Format("{0:F1} ", d / 2.54) + DistanceSuffix;
                        break;
                    case Unit.feet:
                        retStr = String.Format("{0:F2} ", d / 30.48) + DistanceSuffix;
                        break;
                }
            }
            return retStr;
        }

        #if SIMULATION

        //#############################
        // ONLY FOR SIMULATION PURPOSES
        //#############################
        protected void SliderTimer_Tick(object sender, EventArgs e)
        {
            byte rf = GetSliderNumber(ref SIMULATION_DISTANCE);
            //string BASEFOLDER = Application.StartupPath + @"\" + pluginFolder + @"\skins\" + this.currentSkin;
            SIMULATION_DISTANCE--;
            if (SIMULATION_DISTANCE <= 0) SIMULATION_DISTANCE = 254;
            try
            {
                SliderTimer.Stop();
                
                if (this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DETECTFRONT").ToUpper() == "TRUE")
                {
                    if (this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/ENABLECAM").ToUpper() == "TRUE")
                    {
                        // front distance sensors
                        SetBlockImages(rf, rf, rf, rf, true, true);
                        this.CF_updateText("DISTANCE_FRONT", DistanceToString(ref SIMULATION_DISTANCE));
                    }
                    else
                    {
                        SetBlockImages(rf, rf, rf, rf, false, true);
                        this.CF_updateText("DISTANCE_FRONT_NC", DistanceToString(ref SIMULATION_DISTANCE));
                    }
                }
                if (this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DETECTREAR").ToUpper() == "TRUE")
                {
                    if (this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/ENABLECAM").ToUpper() == "TRUE")
                    {
                        // front distance sensors
                        SetBlockImages(rf, rf, rf, rf, true, false);
                        this.CF_updateText("DISTANCE_REAR", DistanceToString(ref SIMULATION_DISTANCE));
                    }
                    else
                    {
                        // front distance sensors
                        SetBlockImages(rf, rf, rf, rf, false, false); 
                        this.CF_updateText("DISTANCE_REAR_NC", DistanceToString(ref SIMULATION_DISTANCE));
                    }
                }
            }
            catch (Exception errmsg) { CFTools.writeError(errmsg.Message, errmsg.StackTrace); }
            
            SliderTimer.Start();
        }
        //#############################
        #endif

        #endregion

        public static string LimitLength (string input, int maxLength)
		{
            string d = string.Empty;

            if (input.Length > maxLength)
			{
				int cBegin, cEnd;
				string sMiddle = "...";
				cBegin = (int)System.Math.Floor(0.5 * maxLength);
				cEnd = maxLength - cBegin - sMiddle.Length;
				d = input.Substring(0, cBegin) + sMiddle + input.Substring(input.Length - cEnd, cEnd);
			}
			else
			{
				d = input;
			}
            return d;
		}

        public override string CF_pluginDefaultConfig()
        {
           string temp = "<APPCONFIG>\r\n" +
                         "    <VERSION>" + Assembly.GetExecutingAssembly().GetName().Version.ToString() + "</VERSION>\r\n" +
                         "    <SETTINGS>\r\n" +
                         "      <DEVICEFRONT>NONE</DEVICEFRONT>\r\n" +
                         "      <DEVICEREAR>NONE</DEVICEREAR>\r\n" +
                         "      <MAINVOLLEVEL>100</MAINVOLLEVEL>\r\n" +
                         "      <USEDEBUG>FALSE</USEDEBUG>\r\n" +
                         "      <DEBUGTOERRORFILE>TRUE</DEBUGTOERRORFILE>\r\n" +
                         "      <DETECTFRONT>FALSE</DETECTFRONT>\r\n" +
                         "      <DETECTREAR>FALSE</DETECTREAR>\r\n" +
                         "      <PAUSEAUDIO>TRUE</PAUSEAUDIO>\r\n" +
                         "      <USEHARDWAREBEEP>FALSE</USEHARDWAREBEEP>\r\n" +
                         "      <ENABLECAM>FALSE</ENABLECAM>\r\n" +
                         "      <VIDEODEVICE>NONE</VIDEODEVICE>\r\n" +
                         "      <DIMENSION>m</DIMENSION>\r\n" +
                         "      <INTERVALLMS>500</INTERVALLMS>\r\n" +
                         "      <HIDETIMEMS>2000</HIDETIMEMS>\r\n" +
                         "      <DISTSHOWREAR>0</DISTSHOWREAR>\r\n" +
                         "      <DISTSHOWFRONT>0</DISTSHOWFRONT>\r\n" +
                         "      <USEVERTICAL>FALSE</USEVERTICAL>\r\n" +
                         "      <HOLDOPENTIME>15</HOLDOPENTIME>\r\n" +
                         "    </SETTINGS>\r\n" +
                         "</APPCONFIG>\r\n";
           return temp;
        }
        private void AddConfigXml()
        {
            WriteLog("AddConfigXml:Start");
            try
            {
#if !(USEPLGPATH)
                // Create applicationdata folder if doesn't exist
                if (!Directory.Exists(CFTools.AppDataPath + "\\Plugins\\" + PluginName))
                {
                    Directory.CreateDirectory(CFTools.AppDataPath + "\\Plugins\\" + PluginName);
                }
#endif
                // Create config.xml file if doesn't exist
                WriteLog("AddConfigXml: Search config file <" + pConfigurationFile + ">");
                if (!File.Exists(pConfigurationFile))
                {
                    // quick creation of new file
                    FileStream fs = new FileStream(pConfigurationFile, FileMode.Create, FileAccess.Write, FileShare.Write);
                    fs.Close();
                    // now insert data to new file
                    StreamWriter sw = new StreamWriter(pConfigurationFile, false, System.Text.Encoding.ASCII);
                    sw.Write(CF_pluginDefaultConfig());
                    sw.Close();
                    WriteLog("AddConfigXml: New config.xml file created!");
                }
            }
            catch (Exception errmsg) { CFTools.writeError(errmsg.Message, errmsg.StackTrace); }
            WriteLog("AddConfigXml:Stop");
            return;
        }
        public static void WriteLog(string msg)
        {
            try
            {
                if (useDebug)
                {
                    if (useErrorOutput)
                        CFTools.writeError(PluginName + ":" + msg);
                    else
                        CFTools.writeModuleLog(PluginName + ":" + msg, LogFilePath);
                }
            }
            catch { }
        } 
        //==========================================================================================
        #region CFPlugin methods
        //==========================================================================================
        public pdc()
		{
            this.CF_params.pauseAudio = false;
            this.CF_params.pluginName = PluginName.ToUpper();
            this.CF_params.isGUI = true;
            this.CF_params.hasSettings = true;
            this.CF_params.hasBasicSettings = true;
		}
        public override void CF_pluginInit()
        {
            try
            {
                this.CF3_initPlugin(PluginName.ToUpper(), true);
                
                useDebug = Boolean.Parse(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/USEDEBUG"));
                useErrorOutput = Boolean.Parse(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DEBUGTOERRORFILE"));

                WriteLog("PDC:Startup");
                WriteLog("CF_pluginInit:Start");
                this.CF_localskinsetup();
                
                //----------------------------------------------------------------
                #region SET DEFAULT VALUES
                // Actions that must execute when the program starts.
                //----------------------------------------------------------------

#if SIMULATION
				SliderTimer = new System.Windows.Forms.Timer();
                SliderTimer.Tick += new EventHandler(SliderTimer_Tick);
                SliderTimer.Interval = 50;
#endif

                #endregion
                
                checkTimer = new System.Windows.Forms.Timer();
                HoldOpenTimer = new System.Windows.Forms.Timer();
                checkTimer.Tick += new EventHandler(checkTimer_Tick);
                HoldOpenTimer.Tick += new EventHandler(HoldOpenTimer_Tick);
                HookHardwareNotifications(this.Handle, false);

#if SIMULATION
                SliderTimer.Start();
#endif
                LoadSettings();
            }
            catch (Exception errmsg) { CFTools.writeError(errmsg.Message, errmsg.StackTrace); }

            finally
            {
                // create event to detect application is loaded and powermode changed
                this.CF_events.applicationLoaded += new EventHandler(PDC_Loaded);
                this.CF_events.powerModeChanged += new Microsoft.Win32.PowerModeChangedEventHandler(PDC_CF_Event_powerModeChanged);
            }
        }

        
        
        public override void CF_pluginShow()
		{
            OpenScreen();
		}
		public override void CF_pluginClose()
		{
            #if SIMULATION
            SliderTimer.Stop();
            #endif

            this.CF_events.powerModeChanged -= new Microsoft.Win32.PowerModeChangedEventHandler(PDC_CF_Event_powerModeChanged);

            // Close the open handles to the device.
            if (deviceRear != null)
            {
                deviceRear.Dispose();
                deviceRear = null;
            }
            if (deviceFront != null)
            {
                deviceFront.Dispose();
                deviceFront = null;
            }
            CloseInterfaces();
            Win32Usb.UnregisterDeviceNotification(this.Handle);
            this.Dispose();
		}	
        public override DialogResult CF_pluginShowSetup()
        {
            WriteLog("CF_pluginShowSetup:Start");

            // This is called by the system when the plugin setup is clicked.
            // Return DialogResult.OK for the main application to update from plugin changes.
            DialogResult returnvalue = DialogResult.Cancel;
            try
            {
                // Creates a new plugin setup instance. If you create a CFDialog or CFSetup you must set 
                // its MainForm property to the main plugins MainForm property.
                setup Setup = new setup(this.MainForm, this.pluginConfig, this.pluginLang);
                returnvalue = Setup.ShowDialog();

                if (returnvalue == DialogResult.OK)
                {
                    // Reloads plugin configuration.
                    this.pluginConfig.Reload();
                    CF_localskinsetup();
                    LoadSettings();
                }
                Setup.Close();
                Setup = null;
            }
            catch (Exception errmsg) { CFTools.writeError(errmsg.Message, errmsg.StackTrace); }
            WriteLog("CF_pluginShowSetup:Stop");
            return returnvalue;
        }
		public override void CF_pluginPause()
		{
			// This is called by the system when it pauses all audio.

		}
		public override void CF_pluginResume()
		{
			// This is called by the system when it resumes all audio.
		}
		public override void CF_pluginCommand(string command, string param1, string param2)
		{
			// Used for plugin to plugin communication.  Parameters can be passed into CF_Main_systemCommands 
			// with CF_Actions.PLUGIN, plugin name, plugin command, and a command parameter.
		}
		public override string CF_pluginData(string command, string param)
		{
			// Used for retrieving information from plugins. You can run CF_getPluginData with a plugin name, 
			// command, and parameter to retrieve information from other plugins running on the system.
			return "";
		}
        public void LoadSettings()
		{
            WriteLog("LoadSettings:Start");
			if (Assembly.GetExecutingAssembly().GetName().Version.ToString().CompareTo(this.pluginConfig.ReadField("/APPCONFIG/VERSION")) != 0)
			{
				this.pluginConfig.WriteField("/APPCONFIG/VERSION", Assembly.GetExecutingAssembly().GetName().Version.ToString());
                this.pluginConfig.Save();
			}
            try { useDebug = Boolean.Parse(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/USEDEBUG")); } 
			catch {useDebug = false;}

            try { GHoldOpenTime = int.Parse(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/HOLDOPENTIME")) * 1000; }
            catch { GHoldOpenTime = 5000; }

            try
            {
                switch (this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DIMENSION"))
                { 
                    case "m": 
                        DistanceUnit = Unit.meters; 
                        DistanceSuffix = " " + this.pluginLang.ReadField("/APPLANG/PDC/DIMENSION/METER").Split(';')[1];
                        break;
                    case "ft": 
                        DistanceUnit = Unit.feet;
                        DistanceSuffix = " " + this.pluginLang.ReadField("/APPLANG/PDC/DIMENSION/FEED").Split(';')[1];                        
                        break;
                    case "inch": 
                        DistanceUnit = Unit.inches;
                        DistanceSuffix = " " + this.pluginLang.ReadField("/APPLANG/PDC/DIMENSION/INCH").Split(';')[1];
                        break;
                }             
            }
            catch { DistanceSuffix = " m"; DistanceUnit = Unit.meters; }

            try
            {
                ENABLE_AUDIO_STOP = bool.Parse(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/PAUSEAUDIO"));
            }
            catch (Exception errmsg) 
            { 
                CFTools.writeError(errmsg.Message, errmsg.StackTrace); 
                ENABLE_AUDIO_STOP = true;
            }

            // The display name is shown in the application to represent
            // the plugin.  This sets the display name from the configuration file.
            this.CF_params.displayName = TranslateText("DISPLAYNAME");
            WriteLog("LoadSettings:Stop");
		}
        private string TranslateText(string text)
        {
            string langread = this.pluginLang.ReadField(LanguageSection + text);

            if (langread == "")
                return text;
            else
                return langread.Replace("\\n", "\n");
        }
        
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                case WM_GRAPHNOTIFY:
                    {
                        HandleGraphEvent();
                        break;
                    }
                case Win32Usb.WM_DEVICECHANGE:
                    {
                       WriteLog("WndProc:ReceiverDevice_OnDeviceChanged");
                       checkTimer.Enabled = true;
                       break;
                    }
            }

            // Pass this message to the video window for notification of system changes
            if (this.videoWindow != null)
                this.videoWindow.NotifyOwnerMessage(m.HWnd, m.Msg, m.WParam, m.LParam);

            base.WndProc(ref m);
        }
        public bool HookHardwareNotifications(IntPtr callback, bool UseWindowHandle)
        {
            try
            {
                Win32Usb.DeviceBroadcastInterface dbdi = new Win32Usb.DeviceBroadcastInterface();
                dbdi.Size = Marshal.SizeOf(dbdi);
                dbdi.Reserved = 0;
                dbdi.DeviceType = Win32Usb.DBT_DEVTYP_DEVICEINTERFACE;
                if (UseWindowHandle)
                {
                    Win32Usb.RegisterDeviceNotification(callback,
                        dbdi,
                        Win32Usb.DEVICE_NOTIFY_ALL_INTERFACE_CLASSES |
                        Win32Usb.DEVICE_NOTIFY_WINDOW_HANDLE);
                }
                else
                {
                    Win32Usb.RegisterDeviceNotification(callback,
                        dbdi,
                        Win32Usb.DEVICE_NOTIFY_ALL_INTERFACE_CLASSES |
                        Win32Usb.DEVICE_NOTIFY_SERVICE_HANDLE);
                }
                return true;
            }
            catch (Exception ex)
            {
                string err = ex.Message;
                return false;
            }
        }
        
        public override void CF_localskinsetup()
		{
			///  This is called to setup the skin.  This will usually be called  in CF_pluginInit.  
			///  It will also called by the system when the resolution has been changed.
            WriteLog("CF_localskinsetup:Start");
            try
            {
                if (Directory.Exists(this.CF_params.pluginSkinPath + @"\FRONT_A"))
                    GLOBAL_SLIDER_PATH = this.CF_params.pluginSkinPath;
                else
                    GLOBAL_SLIDER_PATH = CFTools.StartupPath + @"\" + PluginPath + @"Slider";  
            }
            catch (Exception errmsg) { CFTools.writeError("Error in finding slider path", errmsg.Message); }

            try
			{
                bFullScreen = ((this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DETECTFRONT").ToUpper() == "FALSE") &
                   (this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DETECTREAR").ToUpper() == "FALSE"));

                #region load two possible skins with camera section
                if (this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/ENABLECAM").ToUpper() == "TRUE")
                {
                    if (bFullScreen == false)
                    {
                        CF3_initSection("PDC");
	                    if (bool.Parse(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DETECTFRONT")) == true)
                        {
                            // front distance sensors
                            this.CF3_createPictureBox("FRONT_A");
                            this.CF3_createPictureBox("FRONT_B");
                            this.CF3_createPictureBox("FRONT_C");
                            this.CF3_createPictureBox("FRONT_D");
                            SetBlockImages(20, 20, 20, 20, true, true);
                            this.CF3_createLabel("DISTANCE_FRONT");
                            this.CF_updateText("DISTANCE_FRONT", "-,-- m");
                        }
                        if (bool.Parse(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DETECTREAR")) == true)
                        {
                            // rear distance sensors
                            this.CF3_createPictureBox("REAR_A");
                            this.CF3_createPictureBox("REAR_B");
                            this.CF3_createPictureBox("REAR_C");
                            this.CF3_createPictureBox("REAR_D");
                            SetBlockImages(20, 20, 20, 20, true, false);
                            this.CF3_createLabel("DISTANCE_REAR");
                            this.CF_updateText("DISTANCE_REAR", "-,-- m");
                        }
                        
                        // car symbol/image
                        this.CF_setPictureImage("CAR", Image.FromFile(this.CF_params.pluginSkinPath + @"\car.gif"));
                        this.CF_updateText("TITLE", this.pluginLang.ReadField("/APPLANG/PDC/TITLE"));
                        this.CF_updateText("VERSION", this.pluginLang.ReadField("APPLANG/PDC/VERSIONNAME") + Assembly.GetExecutingAssembly().GetName().Version.ToString());
                    }
                    else
                    {
                        CF3_initSection("PDC_FULL");	
                    }
                    CaptureVideo();
                }
                else
                {
                    CF3_initSection("PDC_NC");
                    if (pdc.VideoDevice != null) CloseInterfaces();
                    if (this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DETECTFRONT").ToUpper() == "TRUE")
                    {
                        // front distance sensors
                        this.CF3_createPictureBox("FRONT_A_NC");
                        this.CF3_createPictureBox("FRONT_B_NC");
                        this.CF3_createPictureBox("FRONT_C_NC");
                        this.CF3_createPictureBox("FRONT_D_NC");
                        SetBlockImages(20, 20, 20, 20, false, true);
                        this.CF3_createLabel("DISTANCE_FRONT_NC");
                        this.CF_updateText("DISTANCE_FRONT_NC", "-,-- m");
                    }
                    if (this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DETECTREAR").ToUpper() == "TRUE")
                    {
                        // rear distance sensors
                        this.CF3_createPictureBox("REAR_A_NC");
                        this.CF3_createPictureBox("REAR_B_NC");
                        this.CF3_createPictureBox("REAR_C_NC");
                        this.CF3_createPictureBox("REAR_D_NC");
                        
                        SetBlockImages(20, 20, 20, 20, false, false);
                        this.CF3_createLabel("DISTANCE_REAR_NC");
                        this.CF_updateText("DISTANCE_REAR_NC", "-,-- m");
                    }
                    this.CF_updateText("TITLE_NC", this.pluginLang.ReadField("/APPLANG/PDC/TITLE"));
                    this.CF_updateText("VERSION_NC", this.pluginLang.ReadField("APPLANG/PDC/VERSIONNAME") + Assembly.GetExecutingAssembly().GetName().Version.ToString());
                    // car symbol/image
                    this.CF_setPictureImage("CAR_NC", Image.FromFile(this.CF_params.pluginSkinPath + @"\car.gif"));
                }
                #endregion
			}
            catch (Exception errmsg) { CFTools.writeError(errmsg.Message, errmsg.StackTrace); }

            WriteLog("CF_localskinsetup:Stop");            
   		}
		#endregion 
		
        public void ShowLargeMessage(string txt)
		{
			
			DialogResult returnvalue = DialogResult.Cancel;
			try
			{
				// Creates a new plugin setup instance. If you create a CFDialog or CFSetup you must set 
				// its MainForm property to the main plugins MainForm property.
                message Msg = new message(this.MainForm, this.pluginConfig, this.pluginLang);
				Msg.MESSAGETEXT = txt;
				Msg.CF_pluginInit();
				returnvalue = Msg.ShowDialog();
				Msg.Close();
				Msg = null;
			}
			catch(Exception errmsg) { CFTools.writeError(errmsg.Message, errmsg.StackTrace); }
		}     
        protected void checkTimer_Tick(object sender, EventArgs e)
		{
		    checkTimer.Stop();
            if (this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DETECTREAR").ToUpper() == "TRUE") GetDataFromDeviceRear();
            if (this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DETECTFRONT").ToUpper() == "TRUE") GetDataFromDeviceFront();    

            //------------------------------------------------------------------
            // Restart the checktimer only if one or both detectors are present
            //------------------------------------------------------------------
            if ((deviceRear != null) || (deviceFront != null)) checkTimer.Start();
		}
        protected void HoldOpenTimer_Tick(object sender, EventArgs e)
        {
            HoldOpenTimer.Stop();
            bCLOSEABLE = true;
        }

        private void OpenDevice(int pID)
        {
            if ((pID == ProductIDRear) & ((this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DETECTREAR").ToUpper() == "TRUE")))
            {
                deviceRear = PDCDevice.FindPDCDevice(VendorID,ProductIDRear);
                if (deviceRear != null)
                {
                    deviceRear.OnDeviceRemoved += new EventHandler(ReceiverDevice_OnDeviceRemoved);
                    deviceRear.OnDeviceArrived += new EventHandler(ReceiverDevice_OnDeviceArrived);
   
                }
            }

            if ((pID == ProductIDFront) & ((this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DETECTFRONT").ToUpper() == "TRUE")))
            {
                deviceFront = PDCDevice.FindPDCDevice(VendorID, ProductIDFront);
                if (deviceFront != null)
                {
                    deviceFront.OnDeviceRemoved += new EventHandler(ReceiverDevice_OnDeviceRemoved);
                    deviceFront.OnDeviceArrived += new EventHandler(ReceiverDevice_OnDeviceArrived);
                }
            }
        }
        private void NoUsb()
		{
            if (this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/ENABLECAM").ToUpper() == "TRUE")
            {
                if (bFullScreen == false)
                {
                    this.CF_setPictureImage("CAR", Image.FromFile(this.CF_params.pluginSkinPath + @"\NoUsb.png"));

                    if (this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DETECTFRONT").ToUpper() == "TRUE")
                    {
                        SetBlockImages(21, 21, 21, 21, true, true);
                        this.CF_updateText("DISTANCE_FRONT", "-,-- " + DistanceSuffix);
                    }
                    if (this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DETECTREAR").ToUpper() == "TRUE")
                    {
                        SetBlockImages(21, 21, 21, 21, true, false);
                        this.CF_updateText("DISTANCE_REAR", "-,-- " + DistanceSuffix);
                    }
                }
            }
            else
            {
                this.CF_setPictureImage("CAR_NC", Image.FromFile(this.CF_params.pluginSkinPath + @"\NoUsb.png"));
                if (this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DETECTFRONT").ToUpper() == "TRUE")
                {
                    SetBlockImages(21, 21, 21, 21, false, true); 
                    this.CF_updateText("DISTANCE_FRONT_NC", "-,-- " + DistanceSuffix);
                }
                if (this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DETECTREAR").ToUpper() == "TRUE")
                {
                    SetBlockImages(21, 21, 21, 21, false, false); 
                    this.CF_updateText("DISTANCE_REAR_NC", "-,-- " + DistanceSuffix);
                }
            }
		}
        //===================================================================================
        // Main section to retrieve data from USB PDC device
        //===================================================================================
        private bool SwitchBuzzer(int pID)
        {
            // switch the buzzer status of the USB driver board
            // send the following byte section to usb device
            // 0x40 = OFF
            // 0x41 = ON
            bool retval = false;

            if ((pID == ProductIDRear) & ((this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DETECTREAR").ToUpper() == "TRUE")))
            {
                try { if (deviceRear == null) OpenDevice(ProductIDRear); }
                finally { }
                if (deviceRear != null)
                {
                    if (outputReport == null) outputReport = new PDCOutputReport(deviceRear);
                        outputReport.Buffer[0] = 0x0;
                        if (BuzzerRearStatus == true)
                        {
                            outputReport.Buffer[1] = 0x40;
                            BuzzerRearStatus = false;
                        }
                        else 
                        {
                            outputReport.Buffer[1] = 0x41;
                            BuzzerRearStatus = true;
                        }
                        retval = BuzzerRearStatus;
                        // Write the data to the device
                        // and secure that this buffer is deleted after the use.
                        //
                        deviceRear.WriteReport(outputReport);
                        outputReport = null;
                }            
            }    
             
            if ((pID == ProductIDFront) & ((this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DETECTFRONT").ToUpper() == "TRUE")))
            {
                try { if (deviceFront == null) OpenDevice(ProductIDFront); }
                finally { }
                if (deviceFront != null)
                {
                    if (outputReport == null) outputReport = new PDCOutputReport(deviceFront);
                        outputReport.Buffer[0] = 0x0;
                        if (BuzzerFrontStatus == true)
                        {
                            outputReport.Buffer[1] = 0x40;
                            BuzzerFrontStatus = false;
                        }
                        else 
                        {
                            outputReport.Buffer[1] = 0x41;
                            BuzzerFrontStatus = true;
                        }
                        retval = BuzzerFrontStatus;
                        // Write the data to the device
                        // and secure that this buffer is deleted after the use.
                        //
                        deviceFront.WriteReport(outputReport);
                        outputReport = null;
                }
            }
            return retval;
        }
        private void CloseScreen()
        {
            if (bCLOSEABLE == true)
                this.Visible = false;
        }
        private void OpenScreen()
        {
            this.Visible = true;
            bCLOSEABLE = false;
            HoldOpenTimer.Interval = GHoldOpenTime;
            HoldOpenTimer.Start(); 
        }
        private void GetDataFromDeviceRear()
		{
            //Sends one byte to the device and reads five bytes back.
            byte mDist; 
            byte mRef;

            // If the device hasn't been detected or it timed out on a previous attempt
            // to access it, look for the device. (currently only rear device)
            try
            {
                if (deviceRear == null) OpenDevice(ProductIDRear);

                // Check if the reverse sensor is active in case of a distance is shorter than 255
                if (byte.TryParse(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DISTSHOWREAR"), out mRef) == false)
                    mRef = 0xff;

                #region deviceRear was found
                if (deviceRear != null)
                {
                    //===========================================================================
                    // The device has been detected, now load the buffer with the byte to be sent
                    //===========================================================================

                    if (outputReport == null) outputReport = new PDCOutputReport(deviceRear);
                    outputReport.Buffer[0] = 0x0;
                    outputReport.Buffer[1] = 0x20;

                    // Write the data to the device
                    // and secure that this buffer is deleted after the use.
                    //
                    deviceRear.WriteReport(outputReport);
                    outputReport = null;

                    // Read the data from the device.
                    //
                    inputReport = deviceRear.ReadReport();
                    if (inputReport == null)
                    {
                        WriteLog("GetDataFromDeviceRear:inputReport read error");
                    }

                    // Get distances         
                    for (int s = 0; s <= 3; s++) Sensor[s].Distance = inputReport.Buffer[s + 1];
                    // Get buzzer status
                    BuzzerRearStatus = Convert.ToBoolean(inputReport.Buffer[5]);
                    
                    // Get the minimum distance detected from the sensors
                    mDist = 0xff;
                    for (int s = 0; s <= 3; s++)
                    {
                        if (Sensor[s].Distance < mDist)
                            mDist = Sensor[s].Distance;
                    }

                    if (useDebug == true)
                    {
                        if ((Sensor[0].OldRearValue != Sensor[0].Distance) ||
                            (Sensor[1].OldRearValue != Sensor[1].Distance) ||
                            (Sensor[2].OldRearValue != Sensor[2].Distance) ||
                            (Sensor[3].OldRearValue != Sensor[3].Distance))
                            {
                                Sensor[0].OldRearValue = Sensor[0].Distance;
                                Sensor[1].OldRearValue = Sensor[1].Distance;
                                Sensor[2].OldRearValue = Sensor[2].Distance;
                                Sensor[3].OldRearValue = Sensor[3].Distance;

                                WriteLog("GetDataFromDeviceRear:Minimal REAR distance: " + mDist.ToString());
                                WriteLog("GetDataFromDeviceRear:Sensor A REAR        : " + Sensor[0].Distance.ToString() + "(" + DistanceToString(ref Sensor[0].Distance) + ")");
                                WriteLog("GetDataFromDeviceRear:Sensor B REAR        : " + Sensor[1].Distance.ToString() + "(" + DistanceToString(ref Sensor[1].Distance) + ")");
                                WriteLog("GetDataFromDeviceRear:Sensor C REAR        : " + Sensor[2].Distance.ToString() + "(" + DistanceToString(ref Sensor[2].Distance) + ")");
                                WriteLog("GetDataFromDeviceRear:Sensor D REAR        : " + Sensor[3].Distance.ToString() + "(" + DistanceToString(ref Sensor[3].Distance) + ")");
                                WriteLog("GetDataFromDeviceRear:-------------------------------");
                            }
                    }

                    if (mDist < mRef)
                    {
                        #region 'DISTANCE <= REFERENCE DISTANCE'
                        
                        // show the plugin main skin ....
                        OpenScreen();

                        STOP_AUDIO = true;
                        // now set the timer to a shorter intervall
                        checkTimer.Interval = 250;

                        // set the audio volume lowering value
                        if (bool.Parse(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/PAUSEAUDIO")) == true) SetReducedAudio();

                        // Quick display update
                        if (bool.Parse(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/ENABLECAM")) == true)
                        {
                            #region SHOW VIDEO FRAME
                            ChangePreviewState(true);
                            this.CF_updateText("DISTANCE_REAR", DistanceToString(ref mDist));
                            // update the distance blocks
                            this.CF_setPictureImage("CAR", Image.FromFile(this.CF_params.pluginSkinPath + @"\car.gif"));
                            SetBlockImages(GetSliderNumber(ref Sensor[3].Distance),
                                            GetSliderNumber(ref Sensor[2].Distance),
                                            GetSliderNumber(ref Sensor[1].Distance),
                                            GetSliderNumber(ref Sensor[0].Distance),
                                            true, false);
                            #endregion SHOW VIDEO FRAME
                        }
                        else
                        {
                            #region DONT SHOW VIDEO FRAME
                            ChangePreviewState(false);
                            this.CF_updateText("DISTANCE_REAR_NC", DistanceToString(ref mDist));
                            this.CF_setPictureImage("CAR_NC", Image.FromFile(this.CF_params.pluginSkinPath + @"\car.gif"));
                            // update the distance blocks without camera
                            SetBlockImages(GetSliderNumber(ref Sensor[3].Distance),
                                            GetSliderNumber(ref Sensor[2].Distance),
                                            GetSliderNumber(ref Sensor[1].Distance),
                                            GetSliderNumber(ref Sensor[0].Distance),
                                            false, false);
                            #endregion DONT SHOW VIDEO FRAME
                        }
                        #endregion 'DISTANCE <= REFERENCE DISTANCE'
                    }
                    else
                    {
                        #region 'DISTANCE > REFERENCE DISTANCE'
                        // now set the timer to a longer intervall
                        checkTimer.Interval = 1000;
                        // If need, restart the audio player
                        
                        STOP_AUDIO = false;

                        if (bool.Parse(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/PAUSEAUDIO")) == true)
                        {
                            SetNormalAudio();
                        }
                        if (bool.Parse(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/ENABLECAM")) == true)
                        {
                            SetBlockImages(21, 21, 21, 21, true, false);
                            this.CF_updateText("DISTANCE_REAR", "-,-- " + DistanceSuffix);
                        }
                        else
                        {
                            SetBlockImages(21, 21, 21, 21, false, false);
                            this.CF_updateText("DISTANCE_REAR_NC", "-,-- " + DistanceSuffix);
                        }

                        if (mRef != 0) CloseScreen();
                        #endregion 'DISTANCE > REFERENCE DISTANCE'
                    }
                }
                #endregion deviceRear was found
                
                else
                {
                    // If the device hasn't been detected set all sensor distances to 255.
                    for (int s = 0; s <= 3; s++)    Sensor[s].Distance = 0xff;
                    if (bool.Parse(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DETECTREAR")) == true) NoUsb();
                }
            }
            catch { }
    }
        private void SetNormalAudio()
        {
            if (VolumeLevel.IsReduced == true)
            {
                this.CF_setMixerLevel((int)CF_MixerDevices.MAIN, 0, (int)VolumeLevel.LastVolume, false);
                VolumeLevel.IsReduced = false;
            }
        }
        private void SetReducedAudio()
        {
            float iRedLevel = 0;

            if (VolumeLevel.IsReduced != true)
            {
                VolumeLevel.IsReduced = true;
                VolumeLevel.LastVolume = this.CF_getMixerLevel((int)CF_MixerDevices.MAIN, 0, false);
                
                try
                {
                    float.TryParse(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/MAINVOLLEVEL"), out iRedLevel);
                    if (iRedLevel > 0)
                    {
                        iRedLevel = VolumeLevel.LastVolume * iRedLevel / 100;
                    }
                }
                catch { }

                this.CF_setMixerLevel((int)CF_MixerDevices.MAIN, 0, (int)iRedLevel, false);
            }
            
        }
        private void GetDataFromDeviceFront()
        {
            //Sends one byte to the device and reads five bytes back.
            byte mDist;
            byte mRef;

            // If the device hasn't been detected or it timed out on a previous attempt
            // to access it, look for the device. (currently only rear device)
            try
            {
                if (deviceFront == null) OpenDevice(ProductIDFront);
                if (deviceFront != null)
                {
                    #region deviceFront was found
                    //===========================================================================
                    // The device has been detected, now load the buffer with the byte to be sent
                    //===========================================================================
                    if (outputReport == null) outputReport = new PDCOutputReport(deviceFront);
                    outputReport.Buffer[0] = 0x0;
                    outputReport.Buffer[1] = 0x20;

                    // Write the data to the device
                    // and secure that this buffer is deleted after the use.
                    //
                    deviceRear.WriteReport(outputReport);
                    outputReport = null;

                    // Read the data from the device.
                    //
                    inputReport = deviceFront.ReadReport();
                    if (inputReport == null)
                    {
                        WriteLog("GetDataFromDeviceFront:inputReport read error");
                    }

                    // Get distances         
                    for (int s = 0; s <= 3; s++) Sensor[s].Distance = inputReport.Buffer[s + 1];
                    // Get buzzer status
                    BuzzerFrontStatus = Convert.ToBoolean(inputReport.Buffer[5]);

                    // Get the minimum distance detected from the sensors
                    mDist = 0xff;
                    for (int s = 0; s <= 3; s++)
                    {
                        if (Sensor[s].Distance < mDist) mDist = Sensor[s].Distance;
                    }

                    // Check if the reverse sensor is active in case of a distance is shorter than 255
                    // Check if the reverse sensor is active in case of a distance is shorter than 255
                    if (byte.TryParse(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DISTSHOWFRONT"), out mRef) == false)
                        mRef = 0xff;

                    #region distance detection started...
                    if (useDebug == true)
                    {
                        if ((Sensor[0].OldFrontValue != Sensor[0].Distance) ||
                            (Sensor[1].OldFrontValue != Sensor[1].Distance) ||
                            (Sensor[2].OldFrontValue != Sensor[2].Distance) ||
                            (Sensor[3].OldFrontValue != Sensor[3].Distance))
                        {
                            Sensor[0].OldFrontValue = Sensor[0].Distance;
                            Sensor[1].OldFrontValue = Sensor[1].Distance;
                            Sensor[2].OldFrontValue = Sensor[2].Distance;
                            Sensor[3].OldFrontValue = Sensor[3].Distance;

                            WriteLog("GetDataFromDeviceFront:Minimal FRONT distance: " + mDist.ToString());
                            WriteLog("GetDataFromDeviceFront:Minimal FRONT distance: " + mDist.ToString());
                            WriteLog("GetDataFromDeviceFront:Sensor A FRONT        : " + Sensor[0].Distance.ToString() + "(" + DistanceToString(ref Sensor[0].Distance) + ")");
                            WriteLog("GetDataFromDeviceFront:Sensor B FRONT        : " + Sensor[1].Distance.ToString() + "(" + DistanceToString(ref Sensor[1].Distance) + ")");
                            WriteLog("GetDataFromDeviceFront:Sensor C FRONT        : " + Sensor[2].Distance.ToString() + "(" + DistanceToString(ref Sensor[2].Distance) + ")");
                            WriteLog("GetDataFromDeviceFront:Sensor D FRONT        : " + Sensor[3].Distance.ToString() + "(" + DistanceToString(ref Sensor[3].Distance) + ")");
                            WriteLog("GetDataFromDeviceFront:-------------------------------");
                        }
                    }

                    if (mDist < mRef)
                    {
                        #region 'DISTANCE <= REFERENCE DISTANCE'
                        // show the plugin main skin ....
                        OpenScreen();
                        STOP_AUDIO = true;
                        // now set the timer to a shorter intervall
                        checkTimer.Interval = 250;

                        // set the audio volume lowering value
                        if (bool.Parse(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/PAUSEAUDIO"))==true) SetReducedAudio();

                        // If need, set the main volume
                        // Quick display update
                        if (bool.Parse(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/ENABLECAM")) == true)
                        {
                            #region SHOW VIDEO FRAME
                            ChangePreviewState(true);
                            this.CF_setPictureImage("CAR", Image.FromFile(this.CF_params.pluginSkinPath + @"\car.gif"));
                            this.CF_updateText("DISTANCE_FRONT", DistanceToString(ref mDist));
                            // update the distance blocks
                            SetBlockImages(GetSliderNumber(ref Sensor[3].Distance),
                                            GetSliderNumber(ref Sensor[2].Distance),
                                            GetSliderNumber(ref Sensor[1].Distance),
                                            GetSliderNumber(ref Sensor[0].Distance),
                                            true, true);
                            #endregion SHOW VIDEO FRAME
                        }
                        else
                        {
                            #region DONT SHOW VIDEO FRAME
                            ChangePreviewState(false);
                            this.CF_setPictureImage("CAR_NC", Image.FromFile(this.CF_params.pluginSkinPath + @"\car.gif"));
                            this.CF_updateText("DISTANCE_FRONT_NC", DistanceToString(ref mDist));

                            // update the distance blocks without camera
                            SetBlockImages(GetSliderNumber(ref Sensor[3].Distance),
                                            GetSliderNumber(ref Sensor[2].Distance),
                                            GetSliderNumber(ref Sensor[1].Distance),
                                            GetSliderNumber(ref Sensor[0].Distance),
                                            false, true);
                            #endregion DONT SHOW VIDEO FRAME
                        }
                        #endregion 'DISTANCE <= REFERENCE DISTANCE'
                    }
                    #endregion
                    else
                    {
                        #region 'DISTANCE > REFERENCE DISTANCE'
                        // now set the timer to a longer intervall
                        checkTimer.Interval = 1000;
                        // If need, restart the audio player
                        STOP_AUDIO = false;
                        if (bool.Parse(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/PAUSEAUDIO")) == true) SetNormalAudio();
                        if (bool.Parse(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/ENABLECAM")) == true)
                        {
                            SetBlockImages(21, 21, 21, 21, true, true);
                            this.CF_updateText("DISTANCE_FRONT", "-,-- " + DistanceSuffix);
                        }
                        else
                        {
                            SetBlockImages(21, 21, 21, 21, false, true);
                            this.CF_updateText("DISTANCE_FRONT_NC", "-,-- " + DistanceSuffix);
                        }
                        if (mRef != 0) CloseScreen();
                        #endregion 'DISTANCE > REFERENCE DISTANCE'
                    }
                    #endregion deviceFront was found
                }
                else
                {
                    #region deviceFront was NOT found
                    // The device hasn't been detected or has been lost.
                    // Set all sensor distances to 255 in case of not finding any sensors
                    for (int s = 0; s <= 3; s++) Sensor[s].Distance = 0xff;
                 
                    if (bool.Parse(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DETECTFRONT")) == true)  NoUsb();
                    #endregion deviceFront was NOT found
                }
            }
            catch { }
        }
        private void SetBlockImages(byte vA, byte vB, byte vC, byte vD, bool useCam, bool IsFront)
        {
            if (IsFront == false)
            {
                if (useCam == true)
                {
                    // display value with camera
                    this.CF_setPictureImage("REAR_A", Image.FromFile(GLOBAL_SLIDER_PATH + @"\REAR_A\SLIDERH_" + Convert.ToString(vA) + ".gif"));
                    this.CF_setPictureImage("REAR_B", Image.FromFile(GLOBAL_SLIDER_PATH + @"\REAR_B\SLIDERH_" + Convert.ToString(vB) + ".gif"));
                    this.CF_setPictureImage("REAR_C", Image.FromFile(GLOBAL_SLIDER_PATH + @"\REAR_C\SLIDERH_" + Convert.ToString(vC) + ".gif"));
                    this.CF_setPictureImage("REAR_D", Image.FromFile(GLOBAL_SLIDER_PATH + @"\REAR_D\SLIDERH_" + Convert.ToString(vD) + ".gif"));
                }
                else
                {
                    // display value without camera
                    this.CF_setPictureImage("REAR_A_NC", Image.FromFile(GLOBAL_SLIDER_PATH + @"\REAR_A\SLIDERH_" + Convert.ToString(vA) + ".gif"));
                    this.CF_setPictureImage("REAR_B_NC", Image.FromFile(GLOBAL_SLIDER_PATH + @"\REAR_B\SLIDERH_" + Convert.ToString(vB) + ".gif"));
                    this.CF_setPictureImage("REAR_C_NC", Image.FromFile(GLOBAL_SLIDER_PATH + @"\REAR_C\SLIDERH_" + Convert.ToString(vC) + ".gif"));
                    this.CF_setPictureImage("REAR_D_NC", Image.FromFile(GLOBAL_SLIDER_PATH + @"\REAR_D\SLIDERH_" + Convert.ToString(vD) + ".gif"));
                }
            }
            else
            {
                if (useCam == true)
                {
                    // display value with camera
                    this.CF_setPictureImage("FRONT_A", Image.FromFile(GLOBAL_SLIDER_PATH + @"\FRONT_A\SLIDERH_" + Convert.ToString(vA) + ".gif"));
                    this.CF_setPictureImage("FRONT_B", Image.FromFile(GLOBAL_SLIDER_PATH + @"\FRONT_B\SLIDERH_" + Convert.ToString(vB) + ".gif"));
                    this.CF_setPictureImage("FRONT_C", Image.FromFile(GLOBAL_SLIDER_PATH + @"\FRONT_C\SLIDERH_" + Convert.ToString(vC) + ".gif"));
                    this.CF_setPictureImage("FRONT_D", Image.FromFile(GLOBAL_SLIDER_PATH + @"\FRONT_D\SLIDERH_" + Convert.ToString(vD) + ".gif"));
                }
                else
                {
                    // display value without camera
                    this.CF_setPictureImage("FRONT_A_NC", Image.FromFile(GLOBAL_SLIDER_PATH + @"\FRONT_A\SLIDERH_" + Convert.ToString(vA) + ".gif"));
                    this.CF_setPictureImage("FRONT_B_NC", Image.FromFile(GLOBAL_SLIDER_PATH + @"\FRONT_B\SLIDERH_" + Convert.ToString(vB) + ".gif"));
                    this.CF_setPictureImage("FRONT_C_NC", Image.FromFile(GLOBAL_SLIDER_PATH + @"\FRONT_C\SLIDERH_" + Convert.ToString(vC) + ".gif"));
                    this.CF_setPictureImage("FRONT_D_NC", Image.FromFile(GLOBAL_SLIDER_PATH + @"\FRONT_D\SLIDERH_" + Convert.ToString(vD) + ".gif"));                
                }
            }
        }
        private void PDC_Loaded(object sender, EventArgs e)
		{
			//===========================================================
			// Event that fires when the aplication is loaded.
			//===========================================================

            WriteLog("PDC_Loaded:Start");

            //----------------------------------------------------------------------------------
            // checkTimer: Timer for detecting the distance to the next obtacle
            //----------------------------------------------------------------------------------
			try 
            {
                checkTimer.Interval = Convert.ToInt16(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/INTERVALLMS")); 
            } 
            catch 
            {
                // standard value = 5 seconds
                checkTimer.Interval = 5000;
            }
            
            //----------------------------------------------------------------------------------
            // HoldOpenTimer: Timer for holding plugin dialog open for n seconds
            //----------------------------------------------------------------------------------
            try
            {
                HoldOpenTimer.Interval = Convert.ToInt16(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/HOLDOPENTIME"));
            }
            catch
            {
                // standard value = 5 seconds
                HoldOpenTimer.Interval = 5000;
            }

            //*******************************************
            // Now search for the PDC USB receiver device
            //*******************************************
            OpenDevice(ProductIDRear);
            OpenDevice(ProductIDFront);

            //*******************************************
            //
            // It is necessary to start the check timer also if no device is found 
            // to ensure that an inserted device could be detected
            //
            checkTimer.Enabled = true;
            			
            if ((deviceRear == null) & (deviceFront == null))
            {
                #if !SIMULATION
                    NoUsb();
                #endif
            }
            WriteLog("PDC_Loaded:Stop");
		}
        private void ReceiverDevice_OnDeviceRemoved(object sender, EventArgs e)
        {
            WriteLog("ReceiverDevice_OnDeviceRemoved:Start");
            if (InvokeRequired)
            {
                Invoke(new EventHandler(ReceiverDevice_OnDeviceRemoved), new object[] { sender, e });
            }
            else
            {
                if (this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DETECTFRONT").ToUpper() == "TRUE")
                {
                    if (deviceFront != null)
                        deviceFront = null;
                }
                if (this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/DETECTREAR").ToUpper() == "TRUE")
                {
                    if (deviceRear != null)
                        deviceRear = null;
                }
                
                #if !SIMULATION
                if ((deviceRear == null) & (deviceFront == null)) 
                    NoUsb(); // this function is started everytime, because of both devices are set to null
                #endif
            }
            SetValues2Zero();
            WriteLog("ReceiverDevice_OnDeviceRemoved:Stop");
        }
        private void ReceiverDevice_OnDeviceArrived(object sender, EventArgs e)
        {
            WriteLog("ReceiverDevice_OnDeviceArrived:Start");
            if (InvokeRequired)
            {
                Invoke(new EventHandler(ReceiverDevice_OnDeviceArrived), new object[] { sender, e });
            }
            else
            {
                WriteLog("ReceiverDevice_OnDeviceArrived:ReceiverDevice_OnDeviceArrived");
                checkTimer.Enabled = true;        
            }
            WriteLog("ReceiverDevice_OnDeviceArrived:Stop");
        }
        private void SetValues2Zero()
        {
            Sensor[0].OldFrontValue = 0;
            Sensor[1].OldFrontValue = 0;
            Sensor[2].OldFrontValue = 0;
            Sensor[3].OldFrontValue = 0;

            Sensor[0].OldRearValue = 0;
            Sensor[1].OldRearValue = 0;
            Sensor[2].OldRearValue = 0;
            Sensor[3].OldRearValue = 0;
        }
        private void PDC_CF_Event_powerModeChanged(object sender, Microsoft.Win32.PowerModeChangedEventArgs e)
        {
            if (e.Mode.ToString() == "Suspend")
            {
                WriteLog("PDC_CF_Event_powerModeChanged - Mode: Suspend");
                checkTimer.Stop();
                HoldOpenTimer.Stop();

                SetValues2Zero();
                
                // NEW: 5.1.2009 - restart video capturing after RESUME 
                if (bool.Parse(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/ENABLECAM")) == true)
                {
                    ChangePreviewState(false);
                    CloseInterfaces();
                }
                
                if (deviceRear != null)
                {
                    deviceRear.Dispose();
                    deviceRear = null;
                }
                if (deviceFront != null)
                {
                    deviceFront.Dispose();
                    deviceFront = null;
                }
            }
            else if (e.Mode.ToString() == "Resume")
            {
                WriteLog("PDC_CF_Event_powerModeChanged - Mode: Resume");

                LoadSettings();
                STOP_AUDIO = false;
                // NEW: 5.1.2009 - restart video capturing after RESUME 
                if (bool.Parse(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/ENABLECAM")) == true )
                {
                    // wait 15 seconds before restart of camera usb device 
                    // there are some devices which takes a little bit longer to restart.

                    int WaitingTime;
                    bool res = int.TryParse(this.pluginConfig.ReadField("/APPCONFIG/SETTINGS/USBDELAYSECONDS"), out WaitingTime);
                    if (res == false)
                        WaitingTime = 15000;
                    else
                    {
                        if ((WaitingTime > 0) && (WaitingTime < 60))
                            WaitingTime *= 1000;
                        else
                            WaitingTime = 15000;
                    }

                    while (WaitingTime > 0)
                    {
                        Application.DoEvents();
                        WaitingTime = WaitingTime - 100;
                        System.Threading.Thread.Sleep(100);
                    }

                    CaptureVideo();
                }

                // Restart PCD devices again if config parameter for a device is set to TRUE
                OpenDevice(ProductIDRear);
                OpenDevice(ProductIDFront);
                if (deviceRear == null) WriteLog("PDC_CF_Event_powerModeChanged:Device REAR IS NULL");
                if (deviceFront == null) WriteLog("PDC_CF_Event_powerModeChanged:Device FRONT IS NULL");
                checkTimer.Start();
                WriteLog("PDC_CF_Event_powerModeChanged:Stop");
            }
        }
  }
}