﻿#include <avr/version.h>

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/eeprom.h>
#include <util/delay.h>
#include <string.h>
#include <stdint.h>


#include "iopins.h"
#include "pinlist.h"
#include "latch.h"

using namespace IO;

#include "ProjectSettings.h"

#include "uart.h"
#include "onewire.h"
#include "ds18x20.h"
#include "servo.h"
#include "Conditions.h"

#ifdef OPTION_LCD_ENABLED
	#include "GraphLCD.h"
#endif

#include "a2d.h"
#include "TempSensors.h"

#include "ProjectPins.h"

#include "Keys.h"

typedef ThreePinLatch<Servo_ShiftRegisterClk_Pin, Servo_ShiftRegisterData_Pin, Servo_ShiftRegisterLatch_Pin, 'R'> ServoDataReg; 
//typedef PinList<Pd2, Pd4, Pd3, Pd5, Pd7, Pb0> ServoDataReg;


#define BAUD 19200
// 2400 for 1MHz and 2MHz internal RC
// #define BAUD 2400


int16_t DestTemp;
int16_t AirDuctDestTemp;
uint8_t volatile VentSpeed;
uint8_t volatile RealVentSpeed;
uint8_t volatile VentDirection;
uint8_t volatile RealVentDirection;
uint8_t volatile Recycling;
uint8_t volatile RealRecycling;
uint8_t volatile Conditioner;
uint8_t volatile PowerState;
uint8_t volatile SomeStateChanged; 

uint8_t volatile CurrTempType;
uint8_t volatile CurrDirType;
uint8_t volatile CurrRecyclingType;
uint8_t volatile CurrVentSpeedType;
uint8_t volatile CurrVentMinSpeedType;
uint8_t volatile CurrConditionIndex;

uint8_t IsUpdateServoMotors; // фхщёЄтєхЄ тю тЁхь  ЁрсюЄ√ т ёхЁтшёэюь Ёхцшьх

uint16_t volatile LastConditionTimer; // ЄрщьхЁ Єюую, ёъюы№ъю тЁхьхэш эх ьхэ хЄё  Єхъє∙хх ёюёЄю эшх. ┬ ёхъєэфрї.

uint8_t volatile ServoPos[5];

#define MAX_VENT_SPEED 8
#define VENT_DIRECTION_BOTTOM 1
#define VENT_DIRECTION_WINDOW 2
#define VENT_DIRECTION_MIDDLE 4

#define MIN_DEST_TEMP 16
#define MAX_DEST_TEMP 32

void CalcRealParameters(void);
uint8_t GetACType(void);


#ifdef OPTION_LCD_ENABLED

	void OutTemp(int16_t* Temp)
	{
	//	char buffer[5];
		uint8_t d1,d2,d3;

		if (*Temp==-100*TEMP_MULTIPLIER)
		{
			d1=10;
			d2=0;
			d3=12;
		}
		else
		if (*Temp==100*TEMP_MULTIPLIER)
		{
			d1=11;
			d2=1;
			d3=12 ;
		}
		else
		{
			int16_t T;
			T =  *Temp/TEMP_MULTIPLIER;
			d2=T % 10;
			d1=T/10;
			T = *Temp % TEMP_MULTIPLIER;
			d3=T;
		}
		LCD_Out_Digits(d1,d2,d3);
	}

void DrawCurrentState(void)
{
	LCD_Out_Man();
	LCD_Out_Vent(1,VentSpeed);
	if (VentDirection==255)
		LCD_Out_Arrows(0,0,0);
	else
		LCD_Out_Arrows(VentDirection&VENT_DIRECTION_WINDOW,VentDirection&VENT_DIRECTION_MIDDLE,VentDirection&VENT_DIRECTION_BOTTOM);

	OutTemp(&DestTemp);
	LCD_Out_Recycling(Recycling);
	LCD_Out_AC(Conditioner);
}

#endif // OPTION_LCD_ENABLED

void ReadACState(void)
{
	uint8_t ch1;
	uint8_t Curr=Conditioner;

	ch1=GetACType();
	if (ch1==AC_TYPE_IN_POSITIVE) 
		Conditioner=AC_Pin::IsSet();
	else
	if (ch1==AC_TYPE_IN_NEGATIVE) 
		Conditioner=(AC_Pin::IsSet()==0);
	else
	if (ch1==AC_TYPE_OFF) 
		Conditioner=false;

	if (Curr!=Conditioner)
		SomeStateChanged=0xFF;

}

void CalcDirectionOnPressed(uint8_t DirPressed)
{
	uint8_t VD=VentDirection;
	if (VD==255)
		VD=DirPressed;
	else
		VD^=DirPressed;
	if (VD!=0)
	{
		uint8_t EnabledDirs=eeprom_read_byte((uint8_t *) (EEPROM_ADDRESS_SERVO+EEPROM_SHIFT_DIRECTIONS_ENABLE));

		if ( (EnabledDirs&(1<<(VD-1)))==0) // если положение является недопустимым, то ищем допустимое
		{
			if (VD&DirPressed)
			{
				for (VD=1;(VD<=7)&& ( !(VD&DirPressed) || !( (EnabledDirs>>(VD-1))&1 ) );VD++);
				if (VD>7)
					VD=0;
			}
			else
			{
				VD=0;
			}
		}
	}
	VentDirection=VD;
}


void ProceedButton(char ch)
{
//	uint8_t Curr=Conditioner;

	switch (ch)
	{
		case KEYS_TEMP_INC_VALUE:
			{
				if (DestTemp==-100*TEMP_MULTIPLIER)
					DestTemp=MIN_DEST_TEMP*TEMP_MULTIPLIER;
				else
				if (DestTemp>=MAX_DEST_TEMP*TEMP_MULTIPLIER)
					DestTemp=100*TEMP_MULTIPLIER;
				else
					DestTemp+=TEMP_MULTIPLIER/2;
				break;
			};
		case KEYS_TEMP_DEC_VALUE:
			{
				if (DestTemp==100*TEMP_MULTIPLIER)
					DestTemp=MAX_DEST_TEMP*TEMP_MULTIPLIER;
				else
				if (DestTemp<=MIN_DEST_TEMP*TEMP_MULTIPLIER)
					DestTemp=-100*TEMP_MULTIPLIER;
				else
					DestTemp-=TEMP_MULTIPLIER/2;break;
			}
		case KEYS_SPEED_INC_VALUE:{
				if (VentSpeed==255)
					VentSpeed=RealVentSpeed;
				else
				if (VentSpeed<MAX_VENT_SPEED)
					VentSpeed++;
				break;
			};
		case KEYS_SPEED_DEC_VALUE:{
				if (VentSpeed==255)
					VentSpeed=RealVentSpeed;
				else
				if (VentSpeed>0)
					VentSpeed--;
				break;
			};

		case KEYS_AUTO_VALUE:{
				VentSpeed=255;
				VentDirection=255;
				break;
			};
		case KEYS_BOTTOM_VALUE:{
			CalcDirectionOnPressed(VENT_DIRECTION_BOTTOM);
/*				if (VentDirection==255)
					VentDirection=VENT_DIRECTION_BOTTOM;
				else
					VentDirection^=VENT_DIRECTION_BOTTOM;*/
				break;
			};
		case KEYS_MIDDLE_VALUE:{
			CalcDirectionOnPressed(VENT_DIRECTION_MIDDLE);
/*				if (VentDirection==255)
					VentDirection=VENT_DIRECTION_MIDDLE;
				else
					VentDirection^=VENT_DIRECTION_MIDDLE;*/
				break;
			};
		case KEYS_WINDOW_VALUE:{
			CalcDirectionOnPressed(VENT_DIRECTION_WINDOW);
/*				if (VentDirection==255)
					VentDirection=VENT_DIRECTION_WINDOW;
				else
					VentDirection^=VENT_DIRECTION_WINDOW;*/
				break;
			};
		case KEYS_RECYCLING_VALUE:{
				Recycling= !Recycling;
				break;
			};
		case  KEYS_POWER_VALUE:{
			uint8_t ch1;
			ch1=GetACType();
			if ((ch1==AC_TYPE_OUT_POSITIVE) || (ch1==AC_TYPE_OUT_NEGATIVE))
				Conditioner=!Conditioner;
			};
	}

	SomeStateChanged=0xFF;
	if (VentDirection==0)
		VentDirection=255;


	CalcRealParameters();
	eeprom_write_block(&DestTemp,(uint8_t *)EEPROM_ADDRESS_CURRENT_STATE,2);
	eeprom_write_byte((uint8_t *)(EEPROM_ADDRESS_CURRENT_STATE+2),VentSpeed);
	eeprom_write_byte((uint8_t *)(EEPROM_ADDRESS_CURRENT_STATE+3),VentDirection);
	eeprom_write_byte((uint8_t *)(EEPROM_ADDRESS_CURRENT_STATE+4),Conditioner);
	eeprom_write_byte((uint8_t *)(EEPROM_ADDRESS_CURRENT_STATE+5),Recycling);
	eeprom_write_byte((uint8_t *)(EEPROM_ADDRESS_CURRENT_STATE+6),PowerState);
}

uint8_t GetACType(void)
{
	return eeprom_read_byte((uint8_t *) (EEPROM_ADDRESS_SERVO+EEPROM_SHIFT_AC_TYPE));
}

uint8_t GetServoType(uint8_t channel)
{
	return eeprom_read_byte((uint8_t *) (EEPROM_ADDRESS_SERVO+EEPROM_SHIFT_SERVO_TYPE+channel));
}

void UpdateServoTypes(void)
{
	uint8_t i;
	uint8_t ch;
	for (i=0;i<SERVO_COUNT;i++)
	{
		ch=GetServoType(i);
		if ((ch==SERVO_TYPE_MOTOR) || (ch==SERVO_TYPE_MOTOR_REVERSE)||(ch==SERVO_TYPE_SWITCH))
			ServoPins::SetConfiguration(ServoPins::In, 1 << i);
		else
			ServoPins::SetConfiguration(ServoPins::Out,1 << i);
	}

	ch=GetACType();
	if ((ch==AC_TYPE_OUT_POSITIVE) || (ch==AC_TYPE_OUT_NEGATIVE))
		AC_Pin::SetDirWrite();
	else
	{
		AC_Pin::SetDirRead();
		AC_Pin::Clear();
	}
}

uint8_t GetRealServoPos(uint8_t channel)
{
		uint8_t ch;
		ch=GetServoType(channel);
		if (ch==SERVO_TYPE_SERVO)
			return ServoPos[channel];
		else
			return ADC_GetState(SERVO_FIRST_ADC_CHANNEL+channel);
}

void SetServoPos(uint8_t ServoNum,uint8_t Pos)
{
	servoSetPosition(ServoNum,Pos);
	ServoPos[ServoNum]=Pos;
}

void UpdateServoMotors(void)
{
	uint8_t i;
	uint8_t j;
	uint8_t ch;
	i=0;
	for (j=0;j<SERVO_COUNT;j++)
	{
		ch=GetServoType(j);
		if (ch==SERVO_TYPE_SERVO)
		{
		}
		else
		if (ch==SERVO_TYPE_SWITCH)
		{
			uint8_t Servo=ServoPos[j];
			if (Servo>127)
				i=i | (1<<(j<<1));
			else
				i=i | (1<<((j<<1)+1));
		}
		else
		if ((ch==SERVO_TYPE_MOTOR)||(ch==SERVO_TYPE_MOTOR_REVERSE))
		{
			int8_t RealServoPos=GetRealServoPos(j);
			int8_t Servo=ServoPos[j];
//			uint8_t Shift;
/*			if (ch==SERVO_TYPE_MOTOR)
				Shift=0;
			else
				Shift=1;*/

			if ((int16_t)RealServoPos>(int16_t)Servo+Servo_Delta)
			{
//				i=i | (1<<((j<<1)+Shift));
				if (ch==SERVO_TYPE_MOTOR)
					i=i | (1<<j*2);
				else
					i=i | (1<<(j*2+1));
			}
			else
			if ((int16_t)RealServoPos<(int16_t)Servo-Servo_Delta)
			{
				if (ch==SERVO_TYPE_MOTOR)
					i=i | (1<<(j*2+1));
				else
					i=i | (1<<(j*2));
			}
//			if (RealServoPos<Servo-Servo_Delta)
//				i=i | (1<<( (j<<1)+((Shift+1)&1) ) );
		}
		
	}

	ch=GetACType();
	if (ch==AC_TYPE_OUT_POSITIVE)
		AC_Pin::Set(Conditioner);
	if (ch==AC_TYPE_OUT_NEGATIVE)
		AC_Pin::Set(Conditioner==0);
/*
	ch=eeprom_read_byte((uint8_t *) (EEPROM_ADDRESS_SERVO+EEPROM_SHIFT_SERVO_TYPE+i));
	if (ADCState[1]>ServoPos[0]+Servo_Delta)
		i=i | 1;
	if (ADCState[1]<ServoPos[0]-Servo_Delta)
		i=i | 2;
	if (ADCState[2]>ServoPos[1]+Servo_Delta)
		i=i | 4;
	if (ADCState[2]<ServoPos[1]-Servo_Delta)
		i=i | 8;
	if (ADCState[3]>ServoPos[2]+Servo_Delta)
		i=i | 16;
	if (ADCState[3]<ServoPos[2]-Servo_Delta)
		i=i | 32;
	if (ADCState[4]>ServoPos[3]+Servo_Delta)
		i=i | 64;
	if (ADCState[4]<ServoPos[3]-Servo_Delta)
		i=i | 128;*/
	ServoDataReg::Write(i);
}

void InitServoMotors(void)
{
	ServoDataReg::InitPins();
}

void CalcRealParameters(void)
{
	if (Recycling)
		RealRecycling=Recycling;
	else
	{
		if (CurrRecyclingType==0)
			RealRecycling=0;
		else
		if (CurrRecyclingType==1)
			RealRecycling=1;
		else
		if ((CurrRecyclingType>=11)&&(CurrRecyclingType<=20))
		{
			uint8_t Period=CurrRecyclingType-10;
			if ((LastConditionTimer/60) % (Period*2)<Period)
				RealRecycling=1;
			else
				RealRecycling=0;	
		}
		if (CurrRecyclingType>=21)
		{
			uint8_t RecyclingTime=CurrRecyclingType-20;
			if ((LastConditionTimer/60)<RecyclingTime)
				RealRecycling=1;
			else
				RealRecycling=0;
		}
	}

	
	if (VentSpeed!=255)
		RealVentSpeed=VentSpeed;
	else
	{
//  1. 1.10 - скорость.  11..n - формула. (1 byte)
//  2. Мин (1 byte)
			if (CurrVentSpeedType<=8)
				 RealVentSpeed=CurrVentSpeedType;
			else
			{
				if (CurrVentSpeedType==11)
				{
// Саморегуляция - (abs(Тзаданная-Тулицы)+abs(Тзаданная-Тсалона))*с3
					RealVentSpeed=abs(DestTemp-Temperatures[TS_OUT])/(10*TEMP_MULTIPLIER)+abs(DestTemp-Temperatures[TS_IN])/(2*TEMP_MULTIPLIER);
				}
				else
				if (CurrVentSpeedType==13)
				{
// 	Поддержка стекла чистым - 3+abs(5-Тснаружи)*c4
					RealVentSpeed = 3+ abs( ((int16_t)5*TEMP_MULTIPLIER-Temperatures[TS_OUT])/50);
				}
				if (RealVentSpeed<CurrVentMinSpeedType)
					RealVentSpeed=CurrVentMinSpeedType;					
			}
	}
	if (RealVentSpeed>MAX_VENT_SPEED)
		RealVentSpeed=MAX_VENT_SPEED;

 	if (VentDirection!=255)
 		RealVentDirection=VentDirection;
	else
		RealVentDirection=CurrDirType;

// два коеффициента - задают теплоемкость машины и теплопроводность ее стенок
#define KTermalCapacity 3
#define KTermalConductivity 4


	AirDuctDestTemp=DestTemp+(DestTemp-Temperatures[TS_IN])*KTermalCapacity+(Temperatures[TS_IN]-Temperatures[TS_OUT])/KTermalConductivity;
	
}

void SetTempServo(void)
{
	
}

void FillCurrentOperationTypes(void)
{
	uint16_t BeginAction;
	BeginAction=ActionIndexes[CurrConditionIndex];
	CurrTempType=GetClimatData(BeginAction,TEMPERATURE_SHIFT);
	CurrDirType=GetClimatData(BeginAction,DIRECTION_SHIFT);
	CurrVentSpeedType=GetClimatData(BeginAction,SPEED_SHIFT);
	CurrVentMinSpeedType=GetClimatData(BeginAction,MIN_SPEED_SHIFT);
	CurrRecyclingType=GetClimatData(BeginAction,RECYCLING_SHIFT);
}

void SendCurrentStateToUART(void)
{
	char Header [4] ={'#','C','I','1'};
	char Footer [2] ={13,10};

	uartSendBuffer((char *)&Header,4);
	uartSendBuffer((char *)&DestTemp,2);
	uartSendBuffer((char *)&VentSpeed,1);
	uartSendBuffer((char *)&RealVentSpeed,1);
	uartSendBuffer((char *)&VentDirection,1);
	uartSendBuffer((char *)&RealVentDirection,1);
	uartSendBuffer((char *)&Conditioner,1);
	uartSendBuffer((char *)&Recycling,1);
	uartSendBuffer((char *)&PowerState,1);

	uartSendBuffer((char *)&Temperatures[0],2*MAXSENSORS);

	uartSendBuffer((char *)&CurrTempType,1);
	uartSendBuffer((char *)&CurrDirType,1);
	uartSendBuffer((char *)&CurrRecyclingType,1);
	uartSendBuffer((char *)&CurrVentSpeedType,1);
	uartSendBuffer((char *)&CurrVentMinSpeedType,1);
	uartSendBuffer((char *)&CurrConditionIndex,1);

	uint8_t i;
	i=SERVO_COUNT;
	uartSendBuffer((char *)&i,1);
	for (i=0;i<SERVO_COUNT;i++)
	{
		int8_t RealServoPos=GetRealServoPos(i);
		uartSendBuffer((char *)&RealServoPos,1);
	}

	uartSendBuffer((char *)&Footer,2);

	
//	for (i=0;i<MAXSENSORS;i++)
//		uartSendBuffer((char *)&Temperatures[i],2);
//	uartSendBuffer((char *)&Footer,2);
}


void SendLog(uint8_t i)
{
	uartSendBuffer((char *)&i,1);
}

void SetPWM(char ch)
{
	OCR2B=ch;
}

int main( void )
{	
/*
	SerialIn_Pin::SetDirRead();
	SerialOut_Pin::SetDirWrite();
	uartInit();

	DDRD=0xFF;
	DDRC=0xFF;
	DDRB=0xFF;
	for (;;)
	{
		PORTD=0xFF;
		PORTB=0xFF;
		PORTC=0xFF;
		SendLog (0x55);
		SendLog (PIND);
		SendLog (PINC);
		SendLog (PINB);
		_delay_ms(500);
		PORTD=0;
		PORTB=0;
		PORTC=0;
		SendLog (0xAA);
		SendLog (PIND);
		SendLog (PINC);
		SendLog (PINB);
		_delay_ms(500);
	}

	Pb5::SetDirWrite();
	for (;;)
	{
		Pb5::Clear();
		_delay_ms(500);
		Pb5::Set();
		_delay_ms(500);
	}
*/
//	инициализируем работу UART
	SerialIn_Pin::SetDirRead();
	SerialOut_Pin::SetDirWrite();

	uartInit();
/*
	uint8_t i8=0;
	for (;;)
	{
		
		uartSendByte(i8);
		i8++;
		_delay_ms(30);
	}
*/

	eeprom_read_block(&DestTemp,(uint8_t *)EEPROM_ADDRESS_CURRENT_STATE,2);
	if (DestTemp==-1)
	{
		DestTemp=22*TEMP_MULTIPLIER;
		VentSpeed=255;
		VentDirection=255;
		Recycling=0;
		PowerState=255;
		Conditioner=0xFF;
	}
	else
	{
		VentSpeed=eeprom_read_byte((uint8_t *)(EEPROM_ADDRESS_CURRENT_STATE+2));
		VentDirection=eeprom_read_byte((uint8_t *)(EEPROM_ADDRESS_CURRENT_STATE+3));
		Conditioner=eeprom_read_byte((uint8_t *)(EEPROM_ADDRESS_CURRENT_STATE+4));
		Recycling=eeprom_read_byte((uint8_t *)(EEPROM_ADDRESS_CURRENT_STATE+5));
		PowerState=eeprom_read_byte((uint8_t *)(EEPROM_ADDRESS_CURRENT_STATE+6));
	}


	uint8_t i,j;
	char ch,ch1,ch2;
	ch1=0;


#ifdef OPTION_LCD_ENABLED
	GraphLCD_Init();
#endif // OPTION_LCD_ENABLED

//	Инициируем ШИМ

//	DDRD |=(1<<PIND3);
	Motor_Pin::SetDirWrite();
	Motor_Pin::Clear();


	TCCR2A = (1<<COM2B1)|(1<<WGM20)|(1<<WGM21);// |(1<<WGM21);
	TCCR2B = (1<<CS20);
	OCR2B=0;

// инициализируем все для измерения температуры
//	ow_set_bus(&PIND,&PORTD,&DDRD,PD2);

	LoadIdsFromEeprom();

	MesureTemperatures();

// начинаем работу с базой климата
	FillActionIndexes();

//	инициализируем работу серв
	UpdateServoTypes();

	servoInit();
	InitServoMotors();

//	инициализируем работу клавиатуры
//	Keys_Pin::SetDirRead();
//	Keys_Pin::Clear();
//	Keys_Start(Keys_ADC_Channel);

	KeysStart();

	UpdateServoTypes();
// Рисуем текущее состояние кнопок на экране

#ifdef OPTION_LCD_ENABLED
	DrawCurrentState();
#endif //OPTION_LCD_ENABLED

	uint8_t CurrentMode=0; // 0 - обычный  режим, 1 - режим тестовых установок

	_delay_ms(850);
	MesureTemperatures();

	CurrConditionIndex=GetClimatAction(&Temperatures[0],DestTemp);
	
	FillCurrentOperationTypes();

	CalcRealParameters();

	for (;;)
	{
		if (CurrentMode==0)
// режит обычной работы
		{
			for (j=0;CurrentMode==0;j++)
			{
				ServoPause();
				KeysRescan();
				ADCChannelsRescan();
				ch=KeysGetCurrentButton();
				if ((ch!=ch1)&&(ch!=0xFF))
				{
					ProceedButton(ch);
				}
				ch1=ch;
				ReadACState();

				if (!uartReceiveBufferIsEmpty())
				{
					ch=uartGetByte();
					if ((ch & 0xF0) ==0)
						ProceedButton(ch);
					if (ch==0xf1)
					{
						CurrentMode=0xFF;
#ifdef OPTION_LCD_ENABLED
						GraphLCD_ClrScr();
#endif //OPTION_LCD_ENABLED
						_delay_ms(1000);
						uartSendBuffer((char *)&ch,1);
						IsUpdateServoMotors=1;
					}
					if (ch==0xf0)
					{
						_delay_ms(1000);
						uartSendBuffer((char *)&ch,1);
					}
				}


				if (SomeStateChanged)
				{
					SomeStateChanged=0;
#ifdef OPTION_LCD_ENABLED
					DrawCurrentState();
#endif //OPTION_LCD_ENABLED
					SendCurrentStateToUART();
				}
				
				UpdateServoMotors();


				if (j==50)
				{
// сюда реально попадаем примерно раз в секунду.
					RescanTemperatures();
					uint8_t Index;
					if (CalcClimatActionError(CurrConditionIndex, &Temperatures[0],DestTemp)>3*TEMP_MULTIPLIER)
// проверяем - не ушли ли мы из текущего режима по суммарным ошибкам более, чем на 3 градуса
// если ушли - выбираем текущий режим и начинаем работать с ним.
//					Index=GetClimatAction(&Temperatures[0],DestTemp);
//					if (Index!=CurrConditionIndex)
					{
						Index=GetClimatAction(&Temperatures[0],DestTemp);
						CurrConditionIndex=Index;
						FillCurrentOperationTypes();
						LastConditionTimer=0;
					}
					else
						LastConditionTimer++;


					CalcRealParameters();


	// задаем скорость вентилятора
					{
						if (RealVentSpeed==0)
							ch=0;
						else
							ch=eeprom_read_byte((uint8_t *) (EEPROM_ADDRESS_SERVO+EEPROM_SHIFT_PWM+RealVentSpeed-1));
						SetPWM(ch);
					}

	// задаем сервы направления
					{
						uint8_t i2;
						for (i2=0;i2<3;i2++)
						{
							ch=eeprom_read_byte((uint8_t *) (EEPROM_ADDRESS_SERVO+EEPROM_SHIFT_MAIN_SERVO+(RealVentDirection-1)*EEPROM_SERVO_DATA_COUNT+i2)); // -1 потому, RealVentDirection - битовое поле и начинаются значения с 1. А индекс - с 0
							SetServoPos(i2+2,ch);// сервы направления - с 2 по 4-й каналы включительно
						}
					}

					if (RealRecycling)
						ch=eeprom_read_byte((uint8_t *) (EEPROM_ADDRESS_SERVO+EEPROM_SHIFT_RECYCLING_SERVO));
					else
						ch=eeprom_read_byte((uint8_t *) (EEPROM_ADDRESS_SERVO+EEPROM_SHIFT_RECYCLING_SERVO+1));
					SetServoPos(1,ch);// серва рециркуляции на 1-м канале


					SendCurrentStateToUART();
					j=0;
				}
				ServoResume();
				WaitForServoCycle();
//				_delay_ms(15);
			}
		}
		else
		{	
			ServoPause();
			ADCChannelsRescan();
			if (!uartReceiveBufferIsEmpty())
			{
				ch=uartGetByte();
				if (ch==0xF0)
				{
					CurrentMode=0;
					_delay_ms(1000);
					uartSendBuffer((char *)&ch,1);
#ifdef OPTION_LCD_ENABLED
					DrawCurrentState();
#endif //OPTION_LCD_ENABLED
				}
				if (ch==0xF2) // грузим данные о сервах
				{
				for(;uartReceiveBufferIsEmpty(););
					j=uartGetByte();
//					eeprom_write_byte((uint8_t *) (EEPROM_ADDRESS_SERVO),j);
					if (j==EEPROM_SERVO_DATA_SIZE)
					{
						for (i=0;i<j;i++)
						{
							for(;uartReceiveBufferIsEmpty(););
							ch=uartGetByte();
							eeprom_write_byte((uint8_t *) (EEPROM_ADDRESS_SERVO+i),ch);
						}
						ch=0xF2;
						uartSendBuffer((char *)&ch,1);
					}
				}
				else
				if (ch==0xF3) // выгружаем данные о сервах
				{
					uartSendBuffer((char *)&ch,1);
					j=EEPROM_SERVO_DATA_SIZE;
					uartSendBuffer((char *)&j,1);
					for (i=0;i<j+1;i++)
					{
						ch=eeprom_read_byte((uint8_t *) (EEPROM_ADDRESS_SERVO+i));
						uartSendBuffer((char *)&ch,1);
					}
					UpdateServoTypes();
				}
				else
				if (ch==0xF4) // Выгружаем данные о датчиках температуры
				{
//					search_sensors();
					cli();
					RescanTemperatures();
					sei();
					_delay_ms(1000);
					cli();
					RescanTemperatures();
					sei();
					uartSendBuffer((char *)&ch,1);
					uartSendBuffer((char *)&SensorCount,1);

					uartSendBuffer((char *)&gSensorIDs[0][0],OW_ROMCODE_SIZE*SensorCount);
					uartSendBuffer((char *)&Temperatures[0],2*SensorCount);

				}
				else
				if (ch==0xF5) // Сканируем датчики температуры
				{
					search_sensors();
					uartSendBuffer((char *)&ch,1);
				}
				else
				if (ch==0xF6) // Загружаем данные о датчиках температуры из епрома
				{
					LoadIdsFromEeprom();
					uartSendBuffer((char *)&ch,1);
				}
				else
				if (ch==0xF7) // Загружаем данные о датчиках температуры из uart в eeprom
				{
					uartSendBuffer((char *)&ch,1);
					for(;uartReceiveBufferIsEmpty(););
					j=uartGetByte();
					SensorCount=j;
					j=j*8;
					for (i=0;i<j;i++)
					{
						for(;uartReceiveBufferIsEmpty(););
						ch=uartGetByte();
						*((uint8_t *)(i+&gSensorIDs[0][0])) =ch;
					}
					SaveIdsToEeprom();
					LoadIdsFromEeprom();
				}
				else
				if (ch==0xF8) // это задает положение сервы. Первый байт - номер сервы. Второй - положение.
				{
					for (;uartReceiveBufferIsEmpty(););
					ch=uartGetByte();
					for (;uartReceiveBufferIsEmpty(););
					ch2=uartGetByte();
					if (ch==0)
					{
						SetPWM(ch2);
					}
					else
					if (ch<6)
					{
						SetServoPos(ch-1,ch2);
					}
					ch=0xF8;
					uartSendBuffer((char *)&ch,1);
				}
				if (ch==0xF9) // заставляет принять Conditions.
				{
					for (;uartReceiveBufferIsEmpty(););
					ch=uartGetByte();
					for (;uartReceiveBufferIsEmpty(););
					ch2=uartGetByte();
					uint16_t Count;
					Count=((uint16_t)ch2<<8) | ch;
//					uartSendBuffer((char *)&Count,2);
					uint16_t Counter=0;
					for (;Count>0;Count--)
					{
						for (;uartReceiveBufferIsEmpty(););
						ch=uartGetByte();
						eeprom_write_byte((uint8_t *) (EEPROM_ADDRESS_CLIMAT_DATA+Counter),ch);
						Counter++;
					}
					ch=0xF9;
					uartSendBuffer((char *)&ch,1);

					FillActionIndexes();
					CurrConditionIndex=GetClimatAction(&Temperatures[0],DestTemp);
					LastConditionTimer=0;
					FillCurrentOperationTypes();

					CalcRealParameters();
				}
				if (ch==0xFA) // заставляет отправить Conditions
				{
					uartSendBuffer((char *)&ch,1);
					uint16_t Size;
					uint16_t LastAction;
					LastAction=ActionIndexes[GetActionCount()-1];
					Size=LastAction+GetActionSize(LastAction)-EEPROM_ADDRESS_CLIMAT_DATA;
					uartSendBuffer((char *)&Size,2);
					uint16_t Pos;
					for (Pos=0;Pos<Size;Pos++)
					{
						ch=eeprom_read_byte((uint8_t *) (EEPROM_ADDRESS_CLIMAT_DATA+Pos));
						uartSendBuffer((char *)&ch,1);

					}
				}
				if (ch==0xFB) // заставляет выдать положения серв
				{
					uartSendBuffer((char *)&ch,1);
					ch=SERVO_COUNT;
					uartSendBuffer((char *)&ch,1);
					for (i=0;i<SERVO_COUNT;i++)
					{
						int8_t RealServoPos=ADC_GetState(SERVO_FIRST_ADC_CHANNEL+i);
						uartSendBuffer((char *)&RealServoPos,1);
					}
				}

				if (ch==0xFC) // Включает и выключает обновление серв. Второй байт 0 - выключает !0 - включает
				{
					uartSendBuffer((char *)&ch,1);
					for (;uartReceiveBufferIsEmpty(););
					ch=uartGetByte();
					uartSendBuffer((char *)&ch,1);
					IsUpdateServoMotors=ch;

				}
				if (ch==0xFD) // вручную задает выходы на сервы
				{
					uartSendBuffer((char *)&ch,1);
					if (SERVO_COUNT<=4)
					{
						for (;uartReceiveBufferIsEmpty(););
						uint8_t ch1=uartGetByte();
						uartSendBuffer((char *)&ch1,1);
						ServoDataReg::Write(ch1);
					}
					else
					{
						for (;uartReceiveBufferIsEmpty(););
						uint8_t ch1=uartGetByte();
						uartSendBuffer((char *)&ch1,1);
						for (;uartReceiveBufferIsEmpty(););
						uint8_t ch2=uartGetByte();
						uartSendBuffer((char *)&ch2,1);
						uint16_t servos=(uint16_t)ch1+(ch2<<8);
						ServoDataReg::Write(servos);
					}



					IsUpdateServoMotors=0;
				}
			}
			if (IsUpdateServoMotors)
			{
				UpdateServoMotors();
				ServoResume();
			}
			_delay_ms(10);
		}
	}



	for (;;)
	{
	}
}
