/*! \file servo.c \brief Interrupt-driven RC Servo function library. */
//*****************************************************************************
//
// File Name    : 'servo.c'
// Title        : Interrupt-driven RC Servo function library
// Author       : Pascal Stang - Copyright (C) 2002
// Created      : 7/31/2002
// Revised      : 8/02/2002
// Version      : 1.0
// Target MCU   : Atmel AVR Series
// Editor Tabs  : 4
//
// This code is distributed under the GNU Public License
//      which can be found at http://www.gnu.org/licenses/gpl.txt
//
//*****************************************************************************

#ifndef WIN32

#include <avr/io.h>
#endif

#include "global.h"
#include "servo.h"

#include "iopins.h"
#include "timer.h"
using namespace IO;


// Program ROM constants

// Global variables
// servo channel registers
u16 ServoPosTics;
u16 ServoPeriodTics;
u08 volatile ServoChannel;
uint8_t volatile PauseEnabled;
//ServoChannelType ServoChannels[SERVO_NUM_CHANNELS];

uint8_t ServoChannelDuty[SERVO_NUM_CHANNELS];
// functions

//! initializes software PWM system
void servoInit(void)
{
    u08 channel;
    // disble the timer1 output compare A interrupt
    cbi(TIMSK1, OCIE1A);
    // set the prescaler for timer1
    timer1SetPrescaler(TIMER_CLK_DIV256);
    // attach the software PWM service routine to timer1 output compare A
    timerAttach(TIMER1OUTCOMPAREA_INT, servoService);
    // enable and clear channels
    for(channel=0; channel<SERVO_NUM_CHANNELS; channel++)
    {
        // set minimum position as default
        ServoChannelDuty[channel] = SERVO_MIN;
        // set default port and pins assignments
        //ServoChannels[channel].port = _SFR_IO_ADDR(SERVO_DEFAULT_PORT);
        //ServoChannels[channel].port = (unsigned char)&SERVO_DEFAULT_PORT;
//        ServoChannels[channel].pin = (1<<channel);
        // set channel pin to output
        // THIS IS OBSOLETED BY THE DYNAMIC CHANNEL TO PORT,PIN ASSIGNMENTS
        //outb(SERVODDR, inb(SERVODDR) | (1<<channel));
    }
#define InitPin(Pin) Pin::SetDirWrite(); Pin::Clear();

	#ifdef SERVO_0_PIN
		InitPin(SERVO_0_PIN)
	#endif
	#ifdef SERVO_1_PIN
		InitPin(SERVO_1_PIN)
	#endif
	#ifdef SERVO_2_PIN
		InitPin(SERVO_2_PIN)
	#endif
	#ifdef SERVO_3_PIN
		InitPin(SERVO_3_PIN)
	#endif
	#ifdef SERVO_4_PIN
		InitPin(SERVO_4_PIN)
	#endif
	#ifdef SERVO_5_PIN
		InitPin(SERVO_5_PIN)
	#endif
	#ifdef SERVO_6_PIN
		InitPin(SERVO_6_PIN)
	#endif
	#ifdef SERVO_7_PIN
		InitPin(SERVO_7_PIN)
	#endif

#undef InitPin

/*	SERVO_0_DDR |= 1<<SERVO_0_PIN;
	SERVO_0_PORT&= ~(_BV(SERVO_0_PIN));
	SERVO_1_DDR |= 1<<SERVO_1_PIN;
	SERVO_1_PORT&= ~(_BV(SERVO_1_PIN));
	SERVO_2_DDR |= 1<<SERVO_2_PIN;
	SERVO_2_PORT&= ~(_BV(SERVO_2_PIN));
	SERVO_3_DDR |= 1<<SERVO_3_PIN;
	SERVO_3_PORT&= ~(_BV(SERVO_3_PIN));
	SERVO_4_DDR |= 1<<SERVO_4_PIN;
	SERVO_4_PORT&= ~(_BV(SERVO_4_PIN));
#ifdef SERVO_5_PIN
	SERVO_5_DDR |= 1<<SERVO_5_PIN;
	SERVO_5_PORT&= ~(_BV(SERVO_5_PIN));
#endif
#ifdef SERVO_6_PIN
	SERVO_6_DDR |= 1<<SERVO_6_PIN;
	SERVO_6_PORT&= ~(_BV(SERVO_6_PIN));
#endif*/
    // set PosTics
    ServoPosTics = 0;
    // set PeriodTics
    ServoPeriodTics = SERVO_MAX*9;
    // set initial interrupt time
    u16 OCValue;
    // read in current value of output compare register OCR1A

    OCValue =  inb(OCR1AL);     // read low byte of OCR1A
    OCValue += inb(OCR1AH)<<8;  // read high byte of OCR1A
    // increment OCR1A value by nextTics
    OCValue += ServoPeriodTics; 
    // set future output compare time to this new value
    outb(OCR1AH, (OCValue>>8));         // write high byte
    outb(OCR1AL, (OCValue & 0x00FF));   // write low byte
    // enable the timer1 output compare A interrupt
    sbi(TIMSK1, OCIE1A);
}

//! turns off software PWM system
void servoOff(void)
{
    // disable the timer1 output compare A interrupt
    cbi(TIMSK1, OCIE1A);
    // detach the service routine
    timerDetach(TIMER1OUTCOMPAREA_INT);
}

//! set port and I/O pin for channel
void servoSetChannelIO(u08 channel, u08 port, u08 pin)
{
/*	if (channel < SERVO_NUM_CHANNELS)
	{
    	ServoChannels[channel].port = port;
    	ServoChannels[channel].pin = (1<<(pin&0x07));
	}*/
}

//! set servo position on channel
void servoSetPosition(u08 channel, u08 position)
{
	if (channel < SERVO_NUM_CHANNELS)
	{
	    // input should be between 0 and SERVO_POSITION_MAX
	    u16 pos_scaled;
	    // calculate scaled position
	    pos_scaled = ((u16)position*(SERVO_MAX-SERVO_MIN)/SERVO_POSITION_MAX)+SERVO_MIN;
	    // set position
	    servoSetPositionRaw(channel, pos_scaled);
	}
}

//! get servo position on channel
u08 servoGetPosition(u08 channel)
{
    return (u08)( ((servoGetPositionRaw(channel)-SERVO_MIN)*SERVO_POSITION_MAX)/(SERVO_MAX-SERVO_MIN) );
}

//! set servo position on channel (raw unscaled format)
void servoSetPositionRaw(u08 channel, u16 position)
{
    // bind to limits
    position = MAX(position, SERVO_MIN);
    position = MIN(position, SERVO_MAX);
    // set position
    ServoChannelDuty[channel] = position;
}

//! get servo position on channel (raw unscaled format)
u16 servoGetPositionRaw(u08 channel)
{
    return ServoChannelDuty[channel];
}

void WaitForServoCycle(void)
{
	uint8_t a;
	a=FALSE;
	for(;!a;)
	{
		cli();
		if (PauseEnabled==0)
			a=TRUE;
		sei();
	}
	a=FALSE;
	for(;!a;)
	{
		cli();
		if (PauseEnabled)
			a=TRUE;
		sei();
	}
}

void ServoPause(void)
{
	uint8_t a;
	a=FALSE;
	for(;!a;)
	{
		cli();
		if (PauseEnabled)
		{
			a=TRUE;
		    timer1SetPrescaler(TIMER_CLK_STOP);
		};
		sei();
	}
}

void ServoResume(void)
{
    timer1SetPrescaler(TIMER_CLK_DIV256);
}

void servoService(void)
{
    u16 nextTics;

	#ifdef ss
		nextTics=15;
	#endif

    if(ServoChannel > SERVO_NUM_CHANNELS)   ServoChannel = 0;
//    if(ServoChannel < SERVO_NUM_CHANNELS)
    {
        
		// turn off current channel
		switch (ServoChannel)
		{
			case 0:{
				PauseEnabled=FALSE;
				#ifdef SERVO_0_PIN
					SERVO_0_PIN::Set();
				#endif
				break;
			}
			case 1:{
				#ifdef SERVO_0_PIN
					SERVO_0_PIN::Clear();
				#endif
				#ifdef SERVO_1_PIN
					SERVO_1_PIN::Set();
				#endif
				break;
			}
			case 2:{
				#ifdef SERVO_1_PIN
					SERVO_1_PIN::Clear();
				#endif
				#ifdef SERVO_2_PIN
					SERVO_2_PIN::Set();
				#endif
				break;
			}
			case 3:{
				#ifdef SERVO_2_PIN
					SERVO_2_PIN::Clear();
				#endif
				#ifdef SERVO_3_PIN
					SERVO_3_PIN::Set();
				#endif
				break;
			}
			case 4:{
				#ifdef SERVO_3_PIN
					SERVO_3_PIN::Clear();
				#endif
				#ifdef SERVO_4_PIN
					SERVO_4_PIN::Set();
				#endif
				break;
			}
			case 5:{
				#ifdef SERVO_4_PIN
					SERVO_4_PIN::Clear();
				#endif
				#ifdef SERVO_5_PIN
					SERVO_5_PIN::Set();
				#endif
				break;
			}
			case 6:{
				#ifdef SERVO_5_PIN
					SERVO_5_PIN::Clear();
				#endif
				#ifdef SERVO_6_PIN
					SERVO_6_PIN::Set();
				#endif
				break;
			}
			case 7:{
				#ifdef SERVO_6_PIN
					SERVO_6_PIN::Clear();
				#endif
				#ifdef SERVO_7_PIN
					SERVO_7_PIN::Set();
				#endif
				break;
			}
			case 8:{
				#ifdef SERVO_7_PIN
					SERVO_7_PIN::Clear();
				#endif
				break;
			}
/*			case 0:PauseEnabled=FALSE;		          SERVO_0_PORT|=_BV(SERVO_0_PIN);break;
			case 1:SERVO_0_PORT&= ~(_BV(SERVO_0_PIN));SERVO_1_PORT|=_BV(SERVO_1_PIN);break;
			case 2:SERVO_1_PORT&= ~(_BV(SERVO_1_PIN));SERVO_2_PORT|=_BV(SERVO_2_PIN);break;
			case 3:SERVO_2_PORT&= ~(_BV(SERVO_2_PIN));SERVO_3_PORT|=_BV(SERVO_3_PIN);break;
			case 4:SERVO_3_PORT&= ~(_BV(SERVO_3_PIN));SERVO_4_PORT|=_BV(SERVO_4_PIN);break;
			case 5:SERVO_4_PORT&= ~(_BV(SERVO_4_PIN));SERVO_5_PORT|=_BV(SERVO_5_PIN);break;
			case 6:SERVO_5_PORT&= ~(_BV(SERVO_5_PIN));SERVO_6_PORT|=_BV(SERVO_6_PIN);break;
			case 7:SERVO_6_PORT&= ~(_BV(SERVO_6_PIN));*/
		}
    }
    
    if(ServoChannel != SERVO_NUM_CHANNELS)
    {
        nextTics = ServoChannelDuty[ServoChannel];
	    ServoPosTics += nextTics;
    }
    else //(Channel == SERVO_NUM_CHANNELS)
    {
        nextTics = ServoPeriodTics-ServoPosTics;
		ServoPosTics = 0;
		PauseEnabled = TRUE;
    }



    // schedule next interrupt
    u16 OCValue;
    // read in current value of output compare register OCR1A
    OCValue =  inb(TCNT1L);     // read low byte of OCR1A
    OCValue += inb(TCNT1H)<<8;  // read high byte of OCR1A
//    OCValue =  inb(OCR1AL);     // read low byte of OCR1A
//    OCValue += inb(OCR1AH)<<8;  // read high byte of OCR1A
    // increment OCR1A value by nextTics
    OCValue += nextTics;
//  OCR1A+=nextTics;
    // set future output compare time to this new value
    outb(OCR1AH, (OCValue>>8));         // write high byte
    outb(OCR1AL, (OCValue & 0x00FF));   // write low byte
    // set our new tic position
//    ServoPosTics += nextTics;
//    if(ServoPosTics >= ServoPeriodTics) ServoPosTics = 0;

    ServoChannel++;

/*
    ServoPosTics += nextTics;
    // schedule next interrupt
//    u16 OCValue;
    // read in current value of output compare register OCR1A
    nextTics +=  inb(OCR1AL);     // read low byte of OCR1A
    nextTics += inb(OCR1AH)<<8;  // read high byte of OCR1A
    // increment OCR1A value by nextTics
//    OCValue += nextTics;
//  OCR1A+=nextTics;
    // set future output compare time to this new value
    outb(OCR1AH, (nextTics>>8));         // write high byte
    outb(OCR1AL, (nextTics & 0x00FF));   // write low byte
    // set our new tic position
//    ServoPosTics += nextTics;
    if(ServoPosTics >= ServoPeriodTics) ServoPosTics = 0;*/
}
