#if !defined(__WG12232_H_INCLUDED__)
#define __WG12232_H_INCLUDED__

#include <pins_arduino.h>

#if defined(WGARDUINO_DBPINS)
/* 
 * This mode is not tested. And not implemented (see wg12232.cpp).
 * My Arduino Duemilanove has only 14 pins for digital output 
 * and first seven of them are PORTD of ATmega168/ATmega328p.
 * So I simply used PORTD as 8-bit parallel port. At least this is fast.
 */
#undef WGARDUINO_DBPINS
#else // WGARDUINO_DBPINS

#define WGARDUINO_DBPINS_PWM_TIMERS 3

/**
 * Arduino <--> WG12232 configuration helper.
 * This structure used to describe PWM counters that should be disabled on DB port.
 */
typedef struct WgArduinoPWM {
  volatile uint8_t *reg;       // PWM-timer-to-disable register
  uint8_t           bit;       // PWM-timer-to-disable bit-mask
};
#endif // WGARDUINO_DBPINS

/**
 * Arduino <--> WG12232 configuration structure.
 * Names for pins used from WG12232 datasheet.
 */
typedef struct WgArduinoConf {
#if defined(WGARDUINO_DBPINS)
  uint8_t db0pin;
  uint8_t db1pin;
  uint8_t db2pin;
  uint8_t db3pin;
  uint8_t db4pin;
  uint8_t db5pin;
  uint8_t db6pin;
  uint8_t db7pin;
#else // WGARDUINO_DBPINS
  volatile uint8_t *dbPORT;
  volatile uint8_t *dbDDR;
  volatile uint8_t *dbPIN;
  WgArduinoPWM      pwm_timers[WGARDUINO_DBPINS_PWM_TIMERS];
#endif // WGARDUINO_DBPINS
  uint8_t a0pin;
  uint8_t rwPin;
  uint8_t ePin;
  uint8_t clPin;
  uint8_t cs1pin;
  uint8_t cs2pin;
  volatile uint8_t *pwmTimerPin[WGARDUINO_DBPINS_PWM_TIMERS];
};

extern WgArduinoConf wg_arduino_conf;

/// Chip selection
#define WGLCD_NOCHIP		0
#define WGLCD_CHIP1		1
#define WGLCD_CHIP2		2
#define WGLCD_CHIPBOTH		3

/// WG12232 command codes
#define WGLCD_CMD_RMW		0xE0
#define WGLCD_CMD_END_RMW	0xEE
#define WGLCD_CMD_SOFT_RST	0xE2

#define WGLCD_CMD_DISP_ON	0xAF
#define WGLCD_CMD_DISP_OFF	0xAE

#define WGLCD_CMD_MAP_SEG	0xA0

#define WGLCD_CMD_SET_SDL	0xC0
#define WGLCD_CMD_SEL_PAGE	0xB8
#define WGLCD_CMD_SEL_COL	0

#define WGLCD_DTM_RW_BIT	2
#define WGLCD_DTM_CD_BIT	1

/// Data Transfer Mode
#define WGLCD_DTM_READDISPDATA	WGLCD_DTM_RW_BIT | WGLCD_DTM_CD_BIT
#define WGLCD_DTM_WRITEDISPDATA	WGLCD_DTM_CD_BIT
#define WGLCD_DTM_READSTATUS	WGLCD_DTM_RW_BIT
#define WGLCD_DTM_WRITECMD	0

/// Segment mapping
#define WGLCD_SEGMAP_NORMAL       0
#define WGLCD_SEGMAP_INVERT       1

/**
 * Control class for WG12232 LCD.
 */
class WgLCD : public WgArduinoConf
{
public:
  WgLCD(const WgArduinoConf& conf);
  
  /**
   * Class initialization.
   * It sets interrupt Arduino Timer2 for 2KHz clock on CL pin,
   *    switches display on, clears screen.
   */
  void setup();
  
  /**
   * Chip selection.
   * \param chip WGLCD_NOCHIP, WGLCD_CHIP1, WGLCD_CHIP2 or WGLCD_CHIPBOTH.
   */
  void selectChip(uint8_t chip);
  
  /**
   * Data Transfer Mode selection.
   * \param dtm Data Transfer Mode value.
   */
  void setDTM(uint8_t dtm);
  
  /**
   * To send a command to WG12232.
   * \param cmd WG12232 command code. See datasheet.
   */
  void command(uint8_t cmd);

  /**
   * To start Read/Modify/Write sequence.
   */
  inline void beginRMW()   { command(WGLCD_CMD_RMW);      }

  /**
   * To finish Read/Modify/Write sequence.
   */
  inline void endRMW()     { command(WGLCD_CMD_END_RMW);  }

  /**
   * Software reset command. It's not equivalent of hardware reset.
   */
  inline void softReset()  { command(WGLCD_CMD_SOFT_RST); }

  /**
   * To switch display on for currently selected chip.
   */
  inline void displayOn()  { command(WGLCD_CMD_DISP_ON);  }

  /**
   * To switch display off for currently selected chip.
   */
  inline void displayOff() { command(WGLCD_CMD_DISP_OFF); }

  /**
   * To set segment mapping for currently selected chip.
   * \param seg Segment mapping mode. WGLCD_SEGMAP_NORMAL or WGLCD_SEGMAP_INVERT.
   */
  inline void setMapping(uint8_t seg) { command(WGLCD_CMD_MAP_SEG + seg);  }

  /**
   * To set Start Display Line.
   * \param line Start Display Line number (0..31).
   */
  inline void setSDL(uint8_t line)    {
    m_sdl = line;
    command(WGLCD_CMD_SET_SDL + line); 
  }

  /**
   * To select page. 32 lines are segmented for 4 pages 8 lines each.
   * \param line Page number (0..3).
   */
  inline void selectPage(uint8_t pag)    { command(WGLCD_CMD_SEL_PAGE + pag); }

  /**
   * To select column. 122 columns are controled by 2 chips 61 column each.
   * \param line Column number (0..60).
   */
  inline void selectColumn(uint8_t col)  { command(WGLCD_CMD_SEL_COL + col);  }

  /**
   * To get WG12232 Status Register value. 
   */
  uint8_t status();

  /**
   * To read data byte from LCD memory.
   */
  uint8_t readChar();

  /**
   * Dummy read function. See WG12232 manual.
   */
  inline void dummyRead() { readChar(); }

  /**
   * To write data byte to LCD memory.
   * \param ch Data byte value (0..255).
   */
  void writeChar(uint8_t ch);

  /**
   * To fill whole column of 4 pages with data byte value.
   * \param x Column number (0..122).
   * \param ch Data byte value (0..255).
   */
  void fillColumn(uint8_t x, uint8_t ch);
  
  /**
   * To fill whole row of 122 columns with 1-bit value.
   * \param y Row number (0..31).
   * \param ch 1-bit value (0..1).
   */
  void fillRow(uint8_t y, uint8_t ch);
  
  /**
   * To fill whole screen with data byte value.
   * \param ch Data byte value (0..255).
   */
  void fillScreen(uint8_t ch);
  
  /**
   * To set pixel with 1-bit value.
   * \param ch 1-bit value (0..1).
   */
  void setPixel(uint8_t x, uint8_t y, uint8_t val);
protected:
  uint8_t m_sdl;
public:
  inline static uint8_t chipFromX(uint8_t x) {
    return (x > 60) ? WGLCD_CHIP2 : WGLCD_CHIP1; 
  }
  inline static uint8_t columnFromX(uint8_t x)  {
    return (x > 60) ? x-61 : x;
  }
  uint8_t pageFromY(uint8_t y);
  uint8_t bitFromY(uint8_t y);
private:
  uint8_t setupTimer2(float timeoutFrequency);
};

extern WgLCD wglcd;

#endif // __WG12232_H_INCLUDED__
