#include <avr/interrupt.h>
#include <wiring.h>
#include "wg12232.h"

WgLCD wglcd(wg_arduino_conf);

#define TIMER_CLOCK_FREQ 2000000.0	//2MHz for /8 prescale from 16MHz

uint8_t wgLCD_latency;
uint8_t wgLCD_timerLoadValue;

/**
 *  Timer2 overflow interrupt vector handler
 */
ISR(TIMER2_OVF_vect)
{
  digitalWrite(wglcd.clPin, !digitalRead(wglcd.clPin));		// Toggle the IO pin to the other state.  
  wgLCD_latency = TCNT2;					/* Capture the current timer value. 
								 * This is how much error we have 
								 * due to interrupt latency and the work in this function
								 */
  TCNT2 = wgLCD_latency + wgLCD_timerLoadValue; 		 // Reload the timer and correct for latency.
}

WgLCD::WgLCD(const WgArduinoConf& conf)
{
#if defined(WGARDUINO_DBPINS)
  db0pin = conf.db0pin;
  db1pin = conf.db1pin;
  db2pin = conf.db2pin;
  db3pin = conf.db3pin;
  db4pin = conf.db4pin;
  db5pin = conf.db5pin;
  db6pin = conf.db6pin;
  db7pin = conf.db7pin;
#else // WGARDUINO_DBPINS
  dbPORT = conf.dbPORT;
  dbDDR  = conf.dbDDR;
  dbPIN  = conf.dbPIN;
  for(uint8_t n = 0 ; n < WGARDUINO_DBPINS_PWM_TIMERS; ++n) {
    pwm_timers[n].reg = conf.pwm_timers[n].reg;
    pwm_timers[n].bit = conf.pwm_timers[n].bit;
  }
#endif // WGARDUINO_DBPINS
  a0pin  = conf.a0pin;
  rwPin  = conf.rwPin;
  ePin   = conf.ePin;
  clPin  = conf.clPin;
  cs1pin = conf.cs1pin;
  cs2pin = conf.cs2pin;
}

void WgLCD::setup()
{
  pinMode(a0pin,  OUTPUT);
  pinMode(rwPin,  OUTPUT);
  pinMode(ePin,   OUTPUT);
  pinMode(clPin,  OUTPUT);
  pinMode(cs1pin, OUTPUT);
  pinMode(cs2pin, OUTPUT);
#if !defined(WGARDUINO_DBPINS)
  for(uint8_t n = 0 ; n < WGARDUINO_DBPINS_PWM_TIMERS; ++n)		// Disable PWM timers
    *(pwm_timers[n].reg) &= pwm_timers[n].bit;
  *dbPORT = 0;							// Set DB port pins to Hi-Z
  *dbDDR = 0;
#endif // WGARDUINO_DBPINS

  wgLCD_timerLoadValue = setupTimer2(2000);			// Start CL timer and get the timer reload value.
  selectChip(WGLCD_CHIPBOTH);					// Switch LCD displays on
  displayOn();
  setSDL(0);							// Set start display line
  setMapping(WGLCD_SEGMAP_NORMAL);
  fillScreen(0);						// Clear screen
}

/**
 * Setup Timer2.
 * Configures the ATMegay168 8-Bit Timer2 to generate an interrupt at the specified frequency.
 * Returns the time load value which must be loaded into TCNT2 inside your ISR routine.
 * See the example usage below.
 */
uint8_t WgLCD::setupTimer2(float timeoutFrequency)
{
  uint8_t result;						/* The value to load into the timer 
								 * to control the timeout interval.
								 */
  result = (uint8_t)((257.0 - (TIMER_CLOCK_FREQ / timeoutFrequency)) + 0.5);
								/* Calculate the timer load value
								 * the 0.5 is for rounding;
								 * The 257 really should be 256
								 * but I get better results with 257,
								 * dont know why.
								 */

  TCCR2A = 0;							// Timer2 Settings: Timer Prescaler /8, mode 0
  TCCR2B = 0 << CS22 | 1 << CS21 | 0 << CS20;			/* Timer clock = 16MHz/8 = 2Mhz or 0.5us
								 * The /8 prescale gives us a good range 
								 * to work with so we just hard code this for now.
								 */
  TIMSK2 = 1 << TOIE2;						// Timer2 Overflow Interrupt Enable
  TCNT2  = result;						// Load the timer for its first cycle
  return result;
}

void WgLCD::selectChip(uint8_t chip)
{
  digitalWrite(cs1pin, (chip & WGLCD_CHIP1) ? LOW : HIGH);
  digitalWrite(cs2pin, (chip & WGLCD_CHIP2) ? LOW : HIGH);
}

void WgLCD::setDTM(uint8_t dtm)
{
  digitalWrite(rwPin, (dtm & WGLCD_DTM_RW_BIT) ? HIGH : LOW); 
  digitalWrite(a0pin, (dtm & WGLCD_DTM_CD_BIT) ? HIGH : LOW);
}

#if !defined(WGARDUINO_DBPINS)
void WgLCD::command(uint8_t cmd)
{
  digitalWrite(ePin,  HIGH); 
  setDTM(WGLCD_DTM_WRITECMD);
  *dbPORT = cmd;
  *dbDDR  = 0xFF;
  digitalWrite(ePin,  LOW);
  *dbPORT = 0;
  *dbDDR  = 0;
}

uint8_t WgLCD::status()
{
  digitalWrite(ePin,  LOW); 
  setDTM(WGLCD_DTM_READSTATUS);
  *dbPORT = 0x00;
  *dbDDR  = 0x00;
  digitalWrite(ePin,  HIGH);
  unsigned char result = *dbPIN;
  return result;
}

uint8_t WgLCD::readChar()
{
  digitalWrite(ePin,  LOW); 
  setDTM(WGLCD_DTM_READDISPDATA);
  *dbPORT = 0x00;
  *dbDDR  = 0x00;
  digitalWrite(ePin,  HIGH);
  unsigned char result = *dbPIN;
  return result;
}

void WgLCD::writeChar(uint8_t ch)
{
  digitalWrite(ePin,  HIGH); 
  setDTM(WGLCD_DTM_WRITEDISPDATA);
  *dbPORT = ch;
  *dbDDR  = 0xFF;
  delayMicroseconds(1);		// Should be __no_operation(); but I didn't test it.
  digitalWrite(ePin,  LOW);
  *dbPORT = 0;
  *dbDDR  = 0;
}
#endif // WGARDUINO_DBPINS

void WgLCD::fillColumn(uint8_t x, uint8_t val)
{
  selectChip(chipFromX(x));
  int cn = columnFromX(x);
  for(uint8_t pn = 0; pn < 4; ++pn) {
    selectPage(pn);
    selectColumn(cn);
    dummyRead();
    writeChar(val);
  }
}

uint8_t WgLCD::pageFromY(uint8_t y)
{
  uint8_t line = m_sdl + y;
  if(line > 31)
    line -= 32;
  return line / 8;
}

uint8_t WgLCD::bitFromY(uint8_t y)
{
  uint8_t line = m_sdl + y;
  if(line > 31)
    line -= 32;
  return line % 8;
}

void WgLCD::fillRow(uint8_t y, uint8_t ch)
{
  uint8_t pixel = 1 << bitFromY(y);
  selectChip(WGLCD_CHIPBOTH);
  selectPage(pageFromY(y));
  selectColumn(0);
  beginRMW();
  for(uint8_t cn = 0; cn < 61; ++cn) {
    dummyRead();
    uint8_t mem   = readChar();
    if(ch)
      writeChar(mem | pixel);
    else
      writeChar(mem & ~pixel);
  }
  endRMW();
}

void WgLCD::fillScreen(uint8_t ch)
{
  selectChip(WGLCD_CHIPBOTH);
  for(uint8_t pn = 0; pn < 4; ++pn) {
    selectPage(pn);
    selectColumn(0);
    beginRMW();
    for(uint8_t cn = 0; cn < 61; ++cn) {
      dummyRead();
      writeChar(ch);
    }
    endRMW();
  }
}

void WgLCD::setPixel(uint8_t x, uint8_t y, uint8_t val)
{
  selectChip(chipFromX(x));
  selectPage(pageFromY(y)); 
  selectColumn(columnFromX(x));
  beginRMW();
  dummyRead();
  uint8_t mem   = readChar();
  uint8_t pixel = 1 << bitFromY(y);
  if(val)
    writeChar(mem | pixel);
  else
    writeChar(mem & ~pixel);
  endRMW();
}

