;	----------------------------------------------------------------------------------------------------	
;	Copyright 2008 William Rouesnel 
;
;	This program is free software: you can redistribute it and/or modify
;    	it under the terms of the GNU General Public License as published by
;    	the Free Software Foundation, either version 3 of the License, or
;    	(at your option) any later version.
;
;    	This program is distributed in the hope that it will be useful,
;    	but WITHOUT ANY WARRANTY; without even the implied warranty of
;    	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;    	GNU General Public License for more details.
;
;    	You should have received a copy of the GNU General Public License
;    	along with this program.  If not, see <http://www.gnu.org/licenses/>.
;	----------------------------------------------------------------------------------------------------	
;
;	IE Bus decoder for the Toyota AVC-LAN
;	By Will Rouesnel <electricitylikesme@hotmail.com>
;
;	Reads IE Bus frames and sends them out as RS-232 output. Will also accept an RS-232 frame and send it
;	over the bus. Actual logic for an IE Bus device must be handled by the USART attached device.
;
;	Timer 0 is used to time normal bits. Timer 1 is used to time start bits. Timer res is 0.2 uS.
;
;	Thanks to Marcin, Louis Frigon and Harry Eaton for the inspiration to actually get this working
;	(in particular Marcin for his design of an easy to build IE Bus driver).
;
;	IEBus Line Driver Quirks
;	Marcin's Line Driver outputs logical 0 for the logic '1' state of the IEBus (Lines < 20mV diff)
;	Hence, logic 1 is actually logic 0 when it is received by the PIC.
;

	LIST P=P16F628A,r=dec,f=INHX8M	; Set PIC16F628A, default radix is decimal (will spec hex)

	#include <p16f628a.inc>

	__CONFIG _CP_OFF & _DATA_CP_OFF & _PWRTE_ON & _WDT_OFF & _LVP_OFF & _MCLRE_OFF & _HS_OSC & _BODEN_ON

;	---------------------------------------------
;	Declarations
;	---------------------------------------------

;	IEBus Constants (format is high and low part of timer, refers to IEBus low times)
IEStartBitH	equ 3
IEStartBitL	equ 85		; 172 uS (if longer then this, likely start bit - +- .8uS)

IEZeroBitH	equ 0
IEZeroBitL	equ 55		; 36 uS = 180, therefore 55 is the extra to make it a zero

IEOneBitH	equ 0
IEOneBitL	equ 99		; 20 uS (factoring timing delays)

IEDeterminateTime	equ	135 ; 27 uS shorter then this is a 1, longer is a 0

BufferLength	equ 85		; Length of the buffer from buffer start (85 bytes)
HelloLength equ 19		; Length of the hello string

IEMaxFrameSize	equ 32	; Maximum size of an IEBus frames data section (32 bytes)

;	----------------------------------------------
; Variables
;	----------------------------------------------
;	IEBus Bit Variables
IENibble		equ	0x20	; Current nibble being filled with IEBus data

; Hello String Variables

HelloPos		equ 0x21	; Memory location for the position tracker of the hello string

; Internal State Variables
IEParity		equ 0x22	; Used to determine the parity of frame elements
IEBitCount		equ 0x23	; Used to count the bits received
IEFlags			equ 0x24	; Bit 0: 0=broadcast,1=point-to-point
IEDataLength	equ 0x25	; Data length

BufferWriteTemp		equ	0x26	; Used to swap between the read address and write address into FSR. FSR is kept read when not in use.
BufferReadAddress	equ 0x27	; Current address from buffer start used to read data to send

;BufferTemp		equ 0x28	; Temp var used for buffer writes << NOT USED ANY MORE

BufferDatasize	equ 0x29	; Stores the amount of data in the buffer atm

BufferStart		equ 0x2A	; Start of the buffer we use to hold RS-232 data (extends to 0x7F)

; Data Area Locations
DA1			equ 0x2		; Data Area 1 PC high location

;	---------------------------------------------
;	OPCODE	DATA		DESCRIPTION	
;	---------------------------------------------
	org		0x0			; RESET vector
	call	Setup		; Setup the PIC and send a hello message to the USART
	goto	ResetProgram	; Loop and wait for start bits

	org		0x4			; INTERRUPT VECTOR
	retfie				; Stop the interrupt (shouldn't have happened but whatever)

ResetProgram
	; Tidying up - new line and carriage return on frame finish
	movlw	'\n'
	call	WriteBuffer
	movlw	'\r'
	call	WriteBuffer
	
	clrf	IENibble
	clrf	IEParity
	clrf	IEFlags
	clrf	IEDataLength
;	---------------------------------------------
Main	; Main program
;	---------------------------------------------
	; Clear Timer 1 ready to time another bit
	clrf	TMR1H
	clrf	TMR1L

WaitForPossibleStartBit
	call	SendData				; Make sure we don't have any data to send
	btfss	PORTB, 0x0				; Wait for Port B, Pin 0 to go high
	goto	WaitForPossibleStartBit	; Loop if PortB is low	
	bsf		T1CON, TMR1ON			; Turn on Timer 1 and time the bit
	call	SendData				; Now is a good time to try sending data as well
WaitForEndOfBit
	btfsc	PORTB, 0x0		; Wait for Port B, Pin 0 to go low
	goto	WaitForEndOfBit	; Loop until Port B is low
	bcf		T1CON, TMR1ON	; Turn off Timer 1

	; Is this a start bit?
	movfw	TMR1H		; Load timer into W
	sublw	IEStartBitH	; Literal - W
	btfss	STATUS, Z	; Is the result 0 ?
	goto	Main		; Definitely not start bit
	movfw	TMR1L		; Move TMR1L to W
	sublw	IEStartBitL	; Literal - W
	btfsc	STATUS, C	; Is the result negative (C=0)?
	goto	Main		; Probably not a start bit
	; Start bits are longer then 172uS
	; Get the broadcast bit
	call	GetBroadcastBit		; Call the function which stores a broadcast bit for us
	call	GetAddressNibble	; Get Master Address
	call	GetAddressNibble	; Get Master Address
	call	GetAddressNibble 	; Get Master Address
	call	GetParityStatus		; Check the parity, output P on parity error
	call	GetAddressNibble	; Get Slave Address
	call	GetAddressNibble	; Get Slave Address
	call	GetAddressNibble 	; Get Slave Address
	call	GetParityStatus		; Check the parity, output P on Parity error
	call	GetAck				; Get the ack bit (this will possibly terminate communications)

	; We must now determine whether there is a terminating nak
	btfss 	STATUS, C			; Ack or Nak?	
	goto	ContinueFrame1		; Continue receiving frame
	btfsc	IEFlags, 0x0		; Nak, do we ignore it (broadcast)
	goto	ResetProgram		; Not broadcast, terminate	
								; Broadcast, ignore it
ContinueFrame1
	call	GetAddressNibble	; Get the configuration bits (1 nibble)
	call	GetParityStatus		; Check the parity, output P on error
	call	GetAck				; Get the ack (terminate if nak)

	; We must now determine whether there is a terminating nak
	btfss 	STATUS, C			; Ack or Nak?	
	goto	ContinueFrame2		; Continue receiving frame
	btfsc	IEFlags, 0x0		; Nak, do we ignore it (broadcast)
	goto	ResetProgram		; Not broadcast, terminate	
								; Broadcast, ignore it
ContinueFrame2
	call	GetMessageLengthNibble1 ; Get the high part of a message length nibble
	call	GetMessageLengthNibble2 ; Get the low part of a message length nibble
	call	GetParityStatus			; Check parity, output P on parity
	call	GetMessageLengthAck		; Get the ack bit AND process the message length field
	
	; We must now determine whether there is a terminating nak
	btfss 	STATUS, C			; Ack or Nak?	
	goto	ContinueFrame3		; Continue receiving frame
	btfsc	IEFlags, 0x0		; Nak, do we ignore it (broadcast)
	goto	ResetProgram		; Not broadcast, terminate	
								; Broadcast, ignore it

ContinueFrame3					; This is where we receive data
	call	GetAddressNibble	; Get Data Nibble High
	call	GetAddressNibble	; Get Data Nibble Low
	call	GetParityStatus		; Check parity
	call	GetDataAck			; Get the ack bit (different for data ack)
	decf	IEDataLength, F		; Decrement IEDataLength		
	btfss	STATUS, Z			; Have all bytes been received?
	goto	ContinueFrame3		; No - continue receiving
								; Yes, loop and reset the program

	goto 	ResetProgram		; Loop and goto the main program
;	---------------------------------------------
GetBroadcastBit ; Receives a broadcast bit
;	---------------------------------------------
	call	GetIEBit			; Get the time of an IEBit
	; See if timer 0 < how long we expect a 0 to go for
	sublw	IEDeterminateTime	; W - IEDeterminateTime
	movlw	'M'					; M greater then
	btfsc	STATUS, C			; Was result negative (C=0)?
	call	WriteUnicast		; C=1, result 0 or positive, this is a 1 - set 'U'
	call	WriteBuffer			; Drop the value into the send buffer
	return

;	---------------------------------------------
WriteUnicast	; Writes a unicast transmission for the broadcast bit
;	---------------------------------------------
	bsf		IEFlags, 0x0		; Indicate unicast
	retlw	'U'					; Return with U

;	---------------------------------------------
GetAddressNibble ; Gets 4 bits of the master address, hex converts and writes it to the buffer
				 ; NOTE: this actually just gets a nibble, but renaming it would kinda pointless
;	---------------------------------------------
	movlw	252			; 255 - 3 so roll over on 4th bit
	movwf	IEBitCount	; Store it
WaitForAddressBitNext	
	bcf		STATUS, C			; Clear C for the bit rotation
	rlf		IENibble, F			; Rotate the bits through anyway (no effect on initial)
	call	GetIEBit	
	; See if timer 0 < how long we expect a 0 to go for
	sublw	IEDeterminateTime	; IEDeterminateTime - W
	btfsc	STATUS, C			; Was result negative (C=0)?
	call	IEBitWriteOne		; Write the bit (need to do parity hence a call)
	incfsz	IEBitCount, F		; Increment IEBitCount, it will roll over on bit 4 so we can write it
	goto	WaitForAddressBitNext	; Wait for the next bit and rotate the bits through IENibble
	movfw	IENibble			; Get the hex in binary
	call	strHex				; Convert it to a ascii hex
	call	WriteBuffer			; Drop the value into the send buffer
	clrf	IENibble			; Make sure we clear IENibble!
	return

;	---------------------------------------------
GetMessageLengthNibble1 ; Gets 4 bits of the data length (the high part)
;	---------------------------------------------
	movlw	252			; 255 - 3 so roll over on 4th bit
	movwf	IEBitCount	; Store it
WaitForMsgLenNext	
	bcf		STATUS, C			; Clear C for the bit rotation
	rlf		IENibble, F			; Rotate the bits through anyway (no effect on initial)
	call	GetIEBit	
	; See if timer 0 < how long we expect a 0 to go for
	sublw	IEDeterminateTime	; IEDeterminateTime - W
	btfsc	STATUS, C			; Was result negative (C=0)?
	call	IEBitWriteOne		; Write the bit (need to do parity hence a call)
	incfsz	IEBitCount, F		; Increment IEBitCount, it will roll over on bit 4 so we can write it
	goto	WaitForMsgLenNext	; Wait for the next bit and rotate the bits through IENibble
	movfw	IENibble			; Get the hex in binary
	movwf	IEDataLength		; Save the high part in the low part
	swapf	IEDataLength, F		; Swap the nibbles to get data length to high
	call	strHex				; Convert it to a ascii hex
	call	WriteBuffer			; Drop the value into the send buffer
	clrf	IENibble			; Make sure we clear IENibble!
	return

;	---------------------------------------------
GetMessageLengthNibble2 ; Gets 4 bits of the data length (the high part)
;	---------------------------------------------
	movlw	252			; 255 - 3 so roll over on 4th bit
	movwf	IEBitCount	; Store it
WaitForMsgLen2Next	
	bcf		STATUS, C			; Clear C for the bit rotation
	rlf		IENibble, F			; Rotate the bits through anyway (no effect on initial)
	call	GetIEBit	
	; See if timer 0 < how long we expect a 0 to go for
	sublw	IEDeterminateTime	; IEDeterminateTime - W
	btfsc	STATUS, C			; Was result negative (C=0)?
	call	IEBitWriteOne		; Write the bit (need to do parity hence a call)
	incfsz	IEBitCount, F		; Increment IEBitCount, it will roll over on bit 4 so we can write it
	goto	WaitForMsgLen2Next	; Wait for the next bit and rotate the bits through IENibble
	movfw	IENibble			; Get the hex in binary
	iorwf	IEDataLength, F		; Superimpose the low nibble into IEDataLength
	call	strHex				; Convert it to a ascii hex
	call	WriteBuffer			; Drop the value into the send buffer
	clrf	IENibble			; Make sure we clear IENibble!
	return

;	---------------------------------------------
GetAck ; Gets the acknowledge bit
;	---------------------------------------------
	call	GetIEBit		
	; See if timer 0 < how long we expect a 0 to go for
	sublw	IEDeterminateTime	; IEDeterminateTime - W
	return

;	---------------------------------------------
GetMessageLengthAck ; Gets the message length ack AND processes the message length field
;	---------------------------------------------
GetMLAckStart
	btfss	PORTB, 0x0	; Wait for input to go high
	goto	GetMLAckStart	; Keep waiting
	clrf	TMR0		; Clear Timer 0
	
	; Process the message length field while waiting on the ack bit
	movfw	IEDataLength		; Get the field
	btfsc	STATUS, Z			; Is this a 0 (i.e. 256 data bytes?)
	goto	SetMaxFrameSize		; 
	sublw	31					; 31 - IEDataLength
	btfsc	STATUS, C			; Test result
	goto	GetMLAckEnd			; Result not negative, no problem
SetMaxFrameSize					; This sets max frame size (used above)
	movlw	IEMaxFrameSize		; Result negative (datalength = 32+) set to exactly 32
	movwf	IEDataLength		; IEDataLength now exactly 32
	
GetMLAckEnd
	btfsc	PORTB, 0x0	; Wait for the input to go low	
	goto	GetMLAckEnd ; Keep waiting
	movfw	TMR0		; Save Timer 0 value	
	
	; See if timer 0 < how long we expect a 0 to go for
	sublw	IEDeterminateTime	; IEDeterminateTime - W
	movlw	0x0					; Set ack
	btfsc	STATUS, C			; Was result negative (C=0)?
	movlw	0x1					; C=1, result 0 or positive, this is a 1 - set nak
	return

;	---------------------------------------------
GetDataAck ; Gets the acknowledge bit of a data byte
;	---------------------------------------------
	call	GetIEBit
	; Save Timer 0 value		
	; See if timer 0 < how long we expect a 0 to go for
	sublw	IEDeterminateTime	; IEDeterminateTime - W
	btfsc	STATUS, C			; Was result negative (C=0)?
	call	DataNak				; This is a NAK (bit = 1), output that
	return

;	---------------------------------------------
DataNak ; Outputs a data nak, checks for broadcast
;	---------------------------------------------
	btfss	IEFlags, 0x0		; Is this a broadcast? (0=yes)
	return						; Yes, return now!
	movlw	'N'					; No - Indicate NAK
	call	WriteBuffer			; Write the buffer
	return

;	---------------------------------------------
GetParityStatus ; Gets the parity bit
;	---------------------------------------------
	call	GetIEBit
	; See if timer 0 < how long we expect a 0 to go for
	sublw	IEDeterminateTime	; IEDeterminateTime - W
	btfsc	STATUS, C			; Was result negative (C=0)?
	comf	IEParity, F			; C=1, this is a 1 (do parity w/ complement)
	movlw	' '					; Put a space in W
	btfsc	IEParity, 0x0		; If this is 1 after parity then parity is bad (should be even)
	movlw	'P'					; Write a P to indicate parity error
	call	WriteBuffer			; Drop the value into the send buffer
	clrf	IEParity			; Make sure we reset IEParity!
	return

;	---------------------------------------------
GetIEBit	; Generic function to time the start and stop of a bit, leaves the TMR0 value in W
;	---------------------------------------------
GetIEBitStart
	btfss	PORTB, 0x0	; Wait for input to go high
	goto	GetIEBitStart	; Keep waiting
	clrf	TMR0		; Clear Timer 0
	call	SendData	; There is minimum 25us delay here, so try and send some data (~1.7us)
GetIEBitEnd
	btfsc	PORTB, 0x0	; Wait for the input to go low	
	goto	GetIEBitEnd ; Keep waiting
	movfw	TMR0		; Save Timer 0 value
	return

;	---------------------------------------------
IEBitWriteOne ; Writes a 1 to the IENibble variable ~0.6 uS
;	---------------------------------------------
	incf	IENibble, F 		; Increment (write the bit)
	comf	IEParity, F			; Mark parity
	return

;	---------------------------------------------
WriteBuffer ; Writes W to the local transmit buffer ~1.7 uS
;	---------------------------------------------
	movwf	INDF				; Write the data (FSR should already be BufferWriteOffset)
	incf	BufferDatasize, F	; Note we have another byte in the buffer

	; Check if we need to wrap the buffer now
	movfw	FSR					; Get the current FSR into W
	incf	FSR, F				; Increment the FSR AFTER we've got it in W
	sublw	0x7F				; Subtract maximal write value from W (old buffer value - end will be 7F)
	movlw	BufferStart			; If we're at the end of the buffer, get ready to shift in start of buffer
	btfsc	STATUS, Z			; If end of buffer, result is Z, reset it - else, do nothing
	movwf	FSR					; Write the location of the start of the buffer
	return

;	---------------------------------------------
SendData	; Sends some data from the buffer ~4.2 uS
;	---------------------------------------------
	; Now might be a good time to try and send some data
	btfss 	PIR1, TXIF			; Is TXREG empty?
	return						; No - return now
	movfw	BufferDatasize		; Test if this is not zero
	btfsc	STATUS, Z			; Is there data?
	return						; No (BufferDataSize is 0)
	movfw	FSR					; Yes. Get the Write address into W
	movwf	BufferWriteTemp		; Store the current write address
	movfw	BufferReadAddress	; Get the BufferRead address
	movwf	FSR					; Access the address location
	movfw	INDF				; Get the data
	movwf	TXREG				; Put it in TXREG
	decf	BufferDatasize, F	; We just sent some data, decrement the buffer
	
	; Now we need to check if we need to wrap around the buffer
	movfw	BufferReadAddress	; Get the old buffer read offset
	incf	BufferReadAddress, F; Increment the address AFTER we get it in W (if not end of buffer then next value is valid)
	sublw	0x7F				; Subtract maximal write value from W
	movlw	BufferStart			; Pre-emptively load the BufferStart point into W (if we need to reset)
	btfsc	STATUS, Z			; Was the result 0?
	movwf	BufferReadAddress	; Yes - Write the location of the start of the buffer (wrap around)
	movfw	BufferWriteTemp		; Get the BufferWrite location
	movwf	FSR					; and put it in the FSR for WriteBuffer
	return						; Done

;	---------------------------------------------
;	Used to convert a nibble to ASCII hex chars ~0.6 uS
;	---------------------------------------------
strHex
	addwf	PCL, F			; Add W to PCL and store in PCL
	dt		"0123456789ABCDEF"

org	0x100	; Put the setup on Page 1
;	---------------------------------------------
Setup	; Initialize serial port, RB0 interrupt, send hello string
;	---------------------------------------------
	; Clear the working memory locations
	clrf	IENibble
	clrf	BufferDatasize
	clrf	TXREG

	; Setup the Buffer Write and Read Offsets
	movlw	BufferStart			; Get the memory address of the buffer start
	movwf	BufferWriteTemp		; Set BufferAddressTemp to BufferStart (this is kind of useless to do really)
	movwf	BufferReadAddress	; Set BufferReadAddress to BufferStart (read address location)
	movwf	FSR					; Set FSR equal to BufferReadOffset
								; ^ This means WriteBuffer assumes FSR is set to the BufferWriteOffset

	; Setup the serial port and periperhal interrupts
	bsf		RCSTA, SPEN
	bsf		INTCON, PEIE	; Need peripherals for the USART interrupt
	; bsf		INTCON, INTE	; Make sure RB0 interrupt is enabled

	; Setup TMR1
	bcf		T1CON, TMR1CS	; Make sure Timer 1 is in timer mode
	
	; SELECT BANK 1
	bsf		STATUS, RP0		; Select bank 1

	bsf		TRISB, 0x0		; set PortB Pin 0 as an input
	bcf		TRISB, 0x3		; set PortB Pin 3 as an output
	bcf		OPTION_REG, T0CS ; Make sure Timer 0 is in timer mode

	; Setup USART
	movlw		0xA				; Set W to 10
	movwf		SPBRG			; Set the SPBRG register to 10 (115.2kbps as a result [approx])
	bsf			TXSTA, BRGH		; Enable high speed baud rate generater
	bcf			TXSTA, SYNC		; Make sure we use asynchronous transmission
	bsf			TXSTA, TXEN		; Enable transmission

	; SELECT BANK 0
	bcf			STATUS, RP0		; Set to memory bank 0
	bsf			PORTB, 0x3		; Set PortB output to high (+5V)	

	; Send the hello message (test RS-232)
	clrf		HelloPos		; Clear the hello string position tracker
SendHelloString
	movlw		DA1				; Load location of Data Area 1
	movwf		PCLATH			; Set PC high location (becomes relevant when we execute a call)
	movfw		HelloPos		; Select the character offset we want
	call		strHello		; Get the character into W
	call		WaitToSend		; Wait till TXREG is clear
	movwf		TXREG			; Load the character into the transmit buffer
	incf		HelloPos, F		; Increment the HelloPos buffer
	movfw		HelloPos		; Load HelloPos into W
	sublw		HelloLength		; Determine if we've sent all chars
	btfss		STATUS, Z		; Is the result zero?
	goto		SendHelloString	; Continue sending the string
	
	; Reset PCLATH to 0 (prevent misdirecting future calls)
	movlw	0x0
	movwf	PCLATH
	
	; Clear the timer before main program runs
	clrf	TMR1L
	clrf	TMR1H

	; Make sure we have a clear interrupt
;	bcf			INTCON, INTF	; Clear the interrupt on PortB
	bsf			INTCON, GIE		; Enable global interrupts
	return

;	---------------------------------------------
WaitToSend	; Loop until
;	---------------------------------------------
	btfss	PIR1, TXIF		; Is TXREG empty?
	goto	WaitToSend		; No
	return
	org		0x200
;	---------------------------------------------
strHello	; Data Area 1 on Code Page 2
;	---------------------------------------------
	addwf	PCL, F
	dt		"IEBus Interpreter\n\r"

	end
